package barcode

import (
	"strings"
	"testing"
)

func init() {
	// Disable trial mode for tests so SAMPLE overlay doesn't interfere.
	SetTrialMode(false)
}

func TestQREncodeBasic(t *testing.T) {
	q := NewQRCode(FormatPNG)
	patt, err := q.GetPattern("HELLO")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if len(patt) == 0 {
		t.Fatal("pattern is empty")
	}
	// Version 1 = 21x21
	if len(patt) != 21 {
		t.Errorf("expected 21x21 pattern for short data, got %dx%d", len(patt), len(patt[0]))
	}
	// Verify it's a square matrix
	for i, row := range patt {
		if len(row) != len(patt) {
			t.Errorf("row %d has length %d, expected %d", i, len(row), len(patt))
		}
	}
	// Verify finder pattern at top-left: (0,0) should be true (dark module)
	if !patt[0][0] {
		t.Error("expected top-left corner (finder pattern) to be dark")
	}
}

func TestQRDrawSVG(t *testing.T) {
	q := NewQRCode(FormatSVG)
	err := q.Draw("Hello World", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := q.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
	if !strings.Contains(svg, "</svg>") {
		t.Error("SVG output missing </svg> tag")
	}
	if !strings.Contains(svg, "<rect") {
		t.Error("SVG output missing <rect elements")
	}
}

func TestQRDrawPNG(t *testing.T) {
	q := NewQRCode(FormatPNG)
	err := q.Draw("Hello World", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := q.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
	// Check PNG magic bytes
	if data[0] != 0x89 || data[1] != 0x50 || data[2] != 0x4E || data[3] != 0x47 {
		t.Error("output is not valid PNG (wrong magic bytes)")
	}
}

func TestQRVersionAuto(t *testing.T) {
	q := NewQRCode(FormatPNG)
	// Short data -> should auto-select version 1 (21x21)
	patt, err := q.GetPattern("AB")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if len(patt) != 21 {
		t.Errorf("expected version 1 (21 modules) for short data, got %d", len(patt))
	}

	// Longer data should auto-select a larger version
	longData := strings.Repeat("A", 100)
	patt2, err := q.GetPattern(longData)
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if len(patt2) <= 21 {
		t.Errorf("expected version > 1 for 100 chars, got %d modules", len(patt2))
	}
}

func TestQRVersionFixed(t *testing.T) {
	q := NewQRCode(FormatPNG)
	q.SetVersion(5)
	// Version 5 = 4*5+17 = 37x37
	patt, err := q.GetPattern("Hi")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if len(patt) != 37 {
		t.Errorf("expected 37 modules for version 5, got %d", len(patt))
	}
}

func TestQREccLevels(t *testing.T) {
	testCases := []struct {
		level int
		name  string
	}{
		{QREccL, "L"},
		{QREccM, "M"},
		{QREccQ, "Q"},
		{QREccH, "H"},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			q := NewQRCode(FormatPNG)
			q.SetErrorCorrectionLevel(tc.level)
			patt, err := q.GetPattern("TEST")
			if err != nil {
				t.Fatalf("GetPattern failed for ECC %s: %v", tc.name, err)
			}
			if len(patt) == 0 {
				t.Fatalf("empty pattern for ECC %s", tc.name)
			}
			// All should produce valid QR codes (square, minimum 21x21)
			if len(patt) < 21 {
				t.Errorf("pattern too small for ECC %s: %d", tc.name, len(patt))
			}
		})
	}
}

func TestQREncodeModes(t *testing.T) {
	t.Run("Numeric", func(t *testing.T) {
		q := NewQRCode(FormatPNG)
		q.SetEncodeMode(QRModeNumeric)
		patt, err := q.GetPattern("0123456789")
		if err != nil {
			t.Fatalf("GetPattern failed: %v", err)
		}
		if len(patt) == 0 {
			t.Fatal("pattern is empty")
		}
		// Numeric mode should produce a valid QR code
		if len(patt) < 21 {
			t.Errorf("pattern too small: %d", len(patt))
		}
	})

	t.Run("AlphaNumeric", func(t *testing.T) {
		q := NewQRCode(FormatPNG)
		q.SetEncodeMode(QRModeAlphaNumeric)
		patt, err := q.GetPattern("HELLO WORLD")
		if err != nil {
			t.Fatalf("GetPattern failed: %v", err)
		}
		if len(patt) == 0 {
			t.Fatal("pattern is empty")
		}
		if len(patt) < 21 {
			t.Errorf("pattern too small: %d", len(patt))
		}
	})

	t.Run("Binary", func(t *testing.T) {
		q := NewQRCode(FormatPNG)
		q.SetEncodeMode(QRModeBinary)
		patt, err := q.GetPattern("Hello World!")
		if err != nil {
			t.Fatalf("GetPattern failed: %v", err)
		}
		if len(patt) == 0 {
			t.Fatal("pattern is empty")
		}
		if len(patt) < 21 {
			t.Errorf("pattern too small: %d", len(patt))
		}
	})
}

func TestQRGetPattern(t *testing.T) {
	q := NewQRCode(FormatPNG)
	patt, err := q.GetPattern("QR")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}

	size := len(patt)
	if size == 0 {
		t.Fatal("pattern is empty")
	}

	// Verify it's a square matrix
	for i := 0; i < size; i++ {
		if len(patt[i]) != size {
			t.Fatalf("row %d: expected length %d, got %d", i, size, len(patt[i]))
		}
	}

	// Verify the three finder patterns exist (7x7 in corners)
	// Top-left finder pattern: rows 0-6, cols 0-6 should form the pattern
	// The outer border should be dark
	for i := 0; i < 7; i++ {
		if !patt[i][0] {
			t.Errorf("top-left finder pattern: patt[%d][0] should be dark", i)
		}
		if !patt[0][i] {
			t.Errorf("top-left finder pattern: patt[0][%d] should be dark", i)
		}
	}

	// Bottom-left finder pattern
	for i := 0; i < 7; i++ {
		if !patt[size-7+i][0] {
			t.Errorf("bottom-left finder pattern: patt[%d][0] should be dark", size-7+i)
		}
	}

	// Top-right finder pattern
	for i := 0; i < 7; i++ {
		if !patt[0][size-7+i] {
			t.Errorf("top-right finder pattern: patt[0][%d] should be dark", size-7+i)
		}
	}
}

func TestQREmptyInput(t *testing.T) {
	q := NewQRCode(FormatPNG)
	err := q.Draw("", 200)
	if err == nil {
		t.Fatal("expected error for empty input")
	}
}

func TestQRDrawSVGFitWidth(t *testing.T) {
	q := NewQRCode(FormatSVG)
	q.SetFitWidth(true)
	err := q.Draw("FitWidth", 300)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := q.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestQRCustomColors(t *testing.T) {
	q := NewQRCode(FormatPNG)
	q.SetForegroundColor(0, 0, 255, 255)
	q.SetBackgroundColor(255, 255, 0, 255)
	err := q.Draw("COLORS", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := q.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
}

func TestQRVersionRangeValidation(t *testing.T) {
	q := NewQRCode(FormatPNG)

	// Valid range
	q.SetVersion(0)
	if q.Version() != 0 {
		t.Errorf("expected version 0, got %d", q.Version())
	}
	q.SetVersion(40)
	if q.Version() != 40 {
		t.Errorf("expected version 40, got %d", q.Version())
	}

	// Invalid range should reset to 0
	q.SetVersion(41)
	if q.Version() != 0 {
		t.Errorf("expected version 0 for invalid 41, got %d", q.Version())
	}
	q.SetVersion(-1)
	if q.Version() != 0 {
		t.Errorf("expected version 0 for invalid -1, got %d", q.Version())
	}
}

func TestQREncodeModeValidation(t *testing.T) {
	q := NewQRCode(FormatPNG)

	q.SetEncodeMode("N")
	if q.EncodeMode() != "N" {
		t.Errorf("expected mode N, got %s", q.EncodeMode())
	}

	q.SetEncodeMode("A")
	if q.EncodeMode() != "A" {
		t.Errorf("expected mode A, got %s", q.EncodeMode())
	}

	// Invalid mode should default to Binary
	q.SetEncodeMode("X")
	if q.EncodeMode() != QRModeBinary {
		t.Errorf("expected mode B for invalid, got %s", q.EncodeMode())
	}
}

func TestQRBase64Output(t *testing.T) {
	q := NewQRCode(FormatPNG)
	err := q.Draw("BASE64", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	b64, err := q.GetImageBase64()
	if err != nil {
		t.Fatalf("GetImageBase64 failed: %v", err)
	}
	if !strings.HasPrefix(b64, "data:image/png;base64,") {
		t.Error("base64 output missing PNG data URI prefix")
	}
}
