package barcode

import (
	"fmt"
	"image"
	"image/color"
	"image/draw"
	"math"
)

// QR Code error correction levels.
const (
	QREccL = 0 // Low (~7%)
	QREccM = 1 // Medium (~15%)
	QREccQ = 2 // Quartile (~25%)
	QREccH = 3 // High (~30%)
)

// QR Code encode modes.
const (
	QRModeNumeric      = "N"
	QRModeAlphaNumeric = "A"
	QRModeBinary       = "B"
	QRModeKanji        = "K"
)

// QRCode is the public QR code encoder.
type QRCode struct {
	BarcodeBase2D
	errorCorrectionLevel int
	version              int    // 0 = auto
	encodeMode           string // "N","A","B","K"
}

// NewQRCode creates a new QR code encoder.
func NewQRCode(outputFormat string) *QRCode {
	q := &QRCode{
		errorCorrectionLevel: QREccM,
		version:              0,
		encodeMode:           QRModeBinary,
	}
	q.InitBase2D(outputFormat)
	return q
}

// SetErrorCorrectionLevel sets the error correction level (QREccL..QREccH).
func (q *QRCode) SetErrorCorrectionLevel(level int) {
	if level >= QREccL && level <= QREccH {
		q.errorCorrectionLevel = level
	}
}

// ErrorCorrectionLevel returns the current error correction level.
func (q *QRCode) ErrorCorrectionLevel() int { return q.errorCorrectionLevel }

// SetVersion sets the QR version (1-40, or 0 for auto).
func (q *QRCode) SetVersion(version int) {
	if version >= 0 && version <= 40 {
		q.version = version
	} else {
		q.version = 0
	}
}

// Version returns the current version setting.
func (q *QRCode) Version() int { return q.version }

// SetEncodeMode sets the encoding mode ("N", "A", "B", "K").
func (q *QRCode) SetEncodeMode(mode string) {
	switch mode {
	case QRModeNumeric, QRModeAlphaNumeric, QRModeBinary, QRModeKanji:
		q.encodeMode = mode
	default:
		q.encodeMode = QRModeBinary
	}
}

// EncodeMode returns the current encode mode.
func (q *QRCode) EncodeMode() string { return q.encodeMode }

// Draw renders the QR code into the internal buffer.
// size specifies both width and height of the output image (square).
func (q *QRCode) Draw(code string, size int) error {
	data := []byte(code)
	if len(data) == 0 {
		return fmt.Errorf("empty input")
	}

	jam := q.createQRJam()
	if err := jam.makeArr(data); err != nil {
		return err
	}
	pattern := jam.patt
	if len(pattern) == 0 {
		return fmt.Errorf("QR encoding produced empty pattern")
	}

	if q.IsSVGOutput() {
		return q.drawSVG(pattern, size, size)
	}
	return q.drawPNG(pattern, size, size)
}

// GetPattern generates and returns the QR code pattern as a 2D bool slice.
func (q *QRCode) GetPattern(code string) ([][]bool, error) {
	data := []byte(code)
	if len(data) == 0 {
		return nil, fmt.Errorf("empty input")
	}

	jam := q.createQRJam()
	if err := jam.makeArr(data); err != nil {
		return nil, err
	}
	return jam.patt, nil
}

func (q *QRCode) createQRJam() *qrJam {
	jam := newQRJam()
	jam.setEncodeMode(q.encodeMode)
	jam.setVersion(q.version)
	jam.setECCRate(q.errorCorrectionLevel)
	return jam
}

func (q *QRCode) drawPNG(pattern [][]bool, width, height int) error {
	patternSize := len(pattern)
	if patternSize == 0 {
		return fmt.Errorf("empty pattern")
	}

	img := image.NewNRGBA(image.Rect(0, 0, width, height))
	draw.Draw(img, img.Bounds(), &image.Uniform{q.backColor}, image.Point{}, draw.Src)

	moduleW := float64(width) / float64(patternSize)
	moduleH := float64(height) / float64(patternSize)

	for r := 0; r < patternSize; r++ {
		for c := 0; c < patternSize; c++ {
			if c < len(pattern) && r < len(pattern[c]) && pattern[c][r] {
				x1 := int(float64(c) * moduleW)
				y1 := int(float64(r) * moduleH)
				x2 := int(float64(c+1) * moduleW)
				y2 := int(float64(r+1) * moduleH)
				fillRect(img, x1, y1, x2, y2, q.foreColor)
			}
		}
	}

	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, width, height)
	}

	return q.encodeImage(img)
}

func (q *QRCode) drawSVG(pattern [][]bool, width, height int) error {
	patternSize := len(pattern)
	if patternSize == 0 {
		return fmt.Errorf("empty pattern")
	}

	q.svgBegin(width, height)
	q.svgRect(0, 0, float64(width), float64(height), q.backColor)

	var moduleW, moduleH float64
	if q.fitWidth {
		moduleW = float64(width) / float64(patternSize)
		moduleH = float64(height) / float64(patternSize)
	} else {
		moduleW = float64(width / patternSize)
		moduleH = float64(height / patternSize)
		if moduleW < 1.0 {
			moduleW = 1.0
		}
		if moduleH < 1.0 {
			moduleH = 1.0
		}
	}

	adjustX := float64(q.pxAdjBlack)
	adjustY := float64(q.pxAdjBlack)

	for r := 0; r < patternSize; r++ {
		for c := 0; c < patternSize; c++ {
			if c < len(pattern) && r < len(pattern[c]) && pattern[c][r] {
				drawX := float64(c) * moduleW
				drawY := float64(r) * moduleH
				drawW := moduleW + adjustX + 0.5
				drawH := moduleH + adjustY + 0.5
				if drawW < 0 {
					drawW = 0
				}
				if drawH < 0 {
					drawH = 0
				}
				q.svgRect(drawX, drawY, drawW, drawH, q.foreColor)
			}
		}
	}

	q.svgEnd()
	return nil
}

// ---------------------------------------------------------------------------
// Internal text types
// ---------------------------------------------------------------------------
const (
	qrTextAuto         = 0
	qrTextNumeric      = 1
	qrTextAlphaNumeric = 2
	qrTextKanji        = 3
	qrTextBinary       = 4
)

// Internal ECC rates
const (
	qrECCLow7     = 0
	qrECCMedium15 = 1
	qrECCQual25   = 2
	qrECCHighQ30  = 3
)

// ---------------------------------------------------------------------------
// qrJam - internal QR code generation engine (port of C++ QRJam / Python _QRJam)
// ---------------------------------------------------------------------------
type qrJam struct {
	lFormatFirstX  [15]int
	lFormatFirstY  [15]int
	lFormatSecondX [15]int
	lFormatSecondY [15]int

	lDeployTableX []int
	lDeployTableY []int
	lDeployOrderX []int
	lDeployOrderY []int
	lRSBlock      []int

	lQRSetVersion  int
	lQRSetECCRate  int
	lQRSetTextType int
	lQRQuitZone    int

	lQRInquireVersion          int
	lQRInquireTextType         int
	lQRInquireNumericBits      int
	lQRInquireAlphaNumericBits int
	lQRInquireKanjiBits        int
	lQRInquireBinaryBits       int
	lQRInquireModuleSize       int

	encodeMode string

	patt [][]bool
}

func newQRJam() *qrJam {
	j := &qrJam{
		lQRSetECCRate:  qrECCMedium15,
		lQRSetTextType: qrTextAuto,
		lQRQuitZone:    4,
		encodeMode:     "Z",
	}
	j.lFormatFirstX = [15]int{0, 1, 2, 3, 4, 5, 7, 8, 8, 8, 8, 8, 8, 8, 8}
	j.lFormatFirstY = [15]int{8, 8, 8, 8, 8, 8, 8, 8, 7, 5, 4, 3, 2, 1, 0}
	return j
}

func (j *qrJam) setEncodeMode(mode string) {
	j.encodeMode = mode
	switch mode {
	case "N":
		j.lQRSetTextType = qrTextNumeric
	case "A":
		j.lQRSetTextType = qrTextAlphaNumeric
	default:
		j.lQRSetTextType = qrTextBinary
	}
}

func (j *qrJam) setVersion(version int) {
	if version >= 0 && version <= 40 {
		j.lQRSetVersion = version
	}
}

func (j *qrJam) setECCRate(rate int) {
	j.lQRSetECCRate = rate
}

// ---------------------------------------------------------------------------
// Lookup tables
// ---------------------------------------------------------------------------

// gDataWords[4][40]
var gDataWords = [4][40]int{
	{19, 34, 55, 80, 108, 68, 78, 97, 116, 68,
		81, 92, 107, 115, 87, 98, 107, 120, 113, 107,
		116, 111, 121, 117, 106, 114, 122, 117, 116, 115,
		115, 115, 115, 115, 121, 121, 122, 122, 117, 118},
	{16, 28, 44, 32, 43, 27, 31, 38, 36, 43,
		50, 36, 37, 40, 41, 45, 46, 43, 44, 41,
		42, 46, 47, 45, 47, 46, 45, 45, 45, 47,
		46, 46, 46, 46, 47, 47, 46, 46, 47, 47},
	{13, 22, 17, 24, 15, 19, 14, 18, 16, 19,
		22, 20, 20, 16, 24, 19, 22, 22, 21, 24,
		22, 24, 24, 24, 24, 22, 23, 24, 23, 24,
		24, 24, 24, 24, 24, 24, 24, 24, 24, 24},
	{9, 16, 13, 9, 11, 15, 13, 14, 12, 15,
		12, 14, 11, 12, 12, 15, 14, 14, 13, 15,
		16, 13, 15, 16, 15, 16, 15, 15, 15, 15,
		15, 15, 15, 16, 15, 15, 15, 15, 15, 15},
}

// gExtraDataWords[4][40]
var gExtraDataWords = [4][40]int{
	{0, 0, 0, 0, 0, 0, 0, 0, 0, 2,
		0, 2, 0, 1, 1, 1, 5, 1, 4, 5,
		4, 7, 5, 4, 4, 2, 4, 10, 7, 10,
		3, 0, 1, 6, 7, 14, 4, 18, 4, 6},
	{0, 0, 0, 0, 0, 0, 0, 2, 2, 1,
		4, 2, 1, 5, 5, 3, 1, 4, 11, 13,
		0, 0, 14, 14, 13, 4, 3, 23, 7, 10,
		29, 23, 21, 23, 26, 34, 14, 32, 7, 31},
	{0, 0, 0, 0, 2, 0, 4, 2, 4, 2,
		4, 6, 4, 5, 7, 2, 15, 1, 4, 5,
		6, 16, 14, 16, 22, 6, 26, 31, 37, 25,
		1, 35, 19, 7, 14, 10, 10, 14, 22, 34},
	{0, 0, 0, 0, 2, 0, 1, 2, 4, 2,
		8, 4, 4, 5, 7, 13, 17, 19, 16, 10,
		6, 0, 14, 2, 13, 4, 28, 31, 26, 25,
		28, 35, 46, 1, 41, 64, 46, 32, 67, 61},
}

// gECCWords[4][40]
var gECCWords = [4][40]int{
	{7, 10, 15, 20, 26, 18, 20, 24, 30, 18,
		20, 24, 26, 30, 22, 24, 28, 30, 28, 28,
		28, 28, 30, 30, 26, 28, 30, 30, 30, 30,
		30, 30, 30, 30, 30, 30, 30, 30, 30, 30},
	{10, 16, 26, 18, 24, 16, 18, 22, 22, 26,
		30, 22, 22, 24, 24, 28, 28, 26, 26, 26,
		26, 28, 28, 28, 28, 28, 28, 28, 28, 28,
		28, 28, 28, 28, 28, 28, 28, 28, 28, 28},
	{13, 22, 18, 26, 18, 24, 18, 22, 20, 24,
		28, 26, 24, 20, 30, 24, 28, 28, 26, 30,
		28, 30, 30, 30, 30, 28, 30, 30, 30, 30,
		30, 30, 30, 30, 30, 30, 30, 30, 30, 30},
	{17, 28, 22, 16, 22, 28, 26, 26, 24, 28,
		24, 28, 22, 24, 24, 30, 28, 28, 26, 28,
		30, 24, 30, 30, 30, 30, 30, 30, 30, 30,
		30, 30, 30, 30, 30, 30, 30, 30, 30, 30},
}

// gRSBlockCount[4][40]
var gRSBlockCount = [4][40]int{
	{1, 1, 1, 1, 1, 2, 2, 2, 2, 4,
		4, 4, 4, 4, 6, 6, 6, 6, 7, 8,
		8, 9, 9, 10, 12, 12, 12, 13, 14, 15,
		16, 17, 18, 19, 19, 20, 21, 22, 24, 25},
	{1, 1, 1, 2, 2, 4, 4, 4, 5, 5,
		5, 8, 9, 9, 10, 10, 11, 13, 14, 16,
		17, 17, 18, 20, 21, 23, 25, 26, 28, 29,
		31, 33, 35, 37, 38, 40, 43, 45, 47, 49},
	{1, 1, 2, 2, 4, 4, 6, 6, 8, 8,
		8, 10, 12, 16, 12, 17, 16, 18, 21, 20,
		23, 23, 25, 27, 29, 34, 34, 35, 38, 40,
		43, 45, 48, 51, 53, 56, 59, 62, 65, 68},
	{1, 1, 2, 4, 4, 4, 5, 6, 8, 8,
		11, 11, 16, 16, 18, 16, 19, 21, 25, 25,
		25, 34, 30, 32, 35, 37, 40, 42, 45, 48,
		51, 54, 57, 60, 63, 66, 70, 74, 77, 81},
}

// Alignment pattern positions by version (1..40)
var qrAlignmentPatterns = map[int][]int{
	1: {6}, 2: {6, 18}, 3: {6, 22}, 4: {6, 26}, 5: {6, 30}, 6: {6, 34},
	7: {6, 22, 38}, 8: {6, 24, 42}, 9: {6, 26, 46}, 10: {6, 28, 50},
	11: {6, 30, 54}, 12: {6, 32, 58}, 13: {6, 34, 62},
	14: {6, 26, 46, 66}, 15: {6, 26, 48, 70}, 16: {6, 26, 50, 74},
	17: {6, 30, 54, 78}, 18: {6, 30, 56, 82}, 19: {6, 30, 58, 86},
	20: {6, 34, 62, 90},
	21: {6, 28, 50, 72, 94}, 22: {6, 26, 50, 74, 98},
	23: {6, 30, 54, 78, 102}, 24: {6, 28, 54, 80, 106},
	25: {6, 32, 58, 84, 110}, 26: {6, 30, 58, 86, 114},
	27: {6, 34, 62, 90, 118},
	28: {6, 26, 50, 74, 98, 122}, 29: {6, 30, 54, 78, 102, 126},
	30: {6, 26, 52, 78, 104, 130}, 31: {6, 30, 56, 82, 108, 134},
	32: {6, 34, 60, 86, 112, 138}, 33: {6, 30, 58, 86, 114, 142},
	34: {6, 34, 62, 90, 118, 146},
	35: {6, 30, 54, 78, 102, 126, 150}, 36: {6, 24, 50, 76, 102, 128, 154},
	37: {6, 28, 54, 80, 106, 132, 158}, 38: {6, 32, 58, 84, 110, 136, 162},
	39: {6, 26, 54, 82, 110, 138, 166}, 40: {6, 30, 58, 86, 114, 142, 170},
}

// Version information bits (versions 7-40)
var qrVersionInfo = [34]int{
	31892, 34236, 39577, 42195, 48118, 51042, 55367, 58893,
	63784, 68472, 70749, 76311, 79154, 84390, 87683, 92361,
	96236, 102084, 102881, 110507, 110734, 117786, 119615, 126325,
	127568, 133589, 136944, 141498, 145311, 150283, 152622, 158308,
	161089, 167017,
}

// GF(256) exp table
var gfExp = [256]int{
	1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38, 76, 152, 45, 90,
	180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192, 157, 39, 78, 156, 37, 74, 148, 53,
	106, 212, 181, 119, 238, 193, 159, 35, 70, 140, 5, 10, 20, 40, 80, 160, 93, 186, 105, 210,
	185, 111, 222, 161, 95, 190, 97, 194, 153, 47, 94, 188, 101, 202, 137, 15, 30, 60, 120, 240,
	253, 231, 211, 187, 107, 214, 177, 127, 254, 225, 223, 163, 91, 182, 113, 226, 217, 175, 67, 134,
	17, 34, 68, 136, 13, 26, 52, 104, 208, 189, 103, 206, 129, 31, 62, 124, 248, 237, 199, 147,
	59, 118, 236, 197, 151, 51, 102, 204, 133, 23, 46, 92, 184, 109, 218, 169, 79, 158, 33, 66,
	132, 21, 42, 84, 168, 77, 154, 41, 82, 164, 85, 170, 73, 146, 57, 114, 228, 213, 183, 115,
	230, 209, 191, 99, 198, 145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219, 171, 75,
	150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25, 50, 100, 200, 141, 7, 14,
	28, 56, 112, 224, 221, 167, 83, 166, 81, 162, 89, 178, 121, 242, 249, 239, 195, 155, 43, 86,
	172, 69, 138, 9, 18, 36, 72, 144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11, 22,
	44, 88, 176, 125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71, 142, 1,
}

// GF(256) log table
var gfLog = [256]int{
	0, 0, 1, 25, 2, 50, 26, 198, 3, 223, 51, 238, 27, 104, 199, 75, 4, 100, 224, 14,
	52, 141, 239, 129, 28, 193, 105, 248, 200, 8, 76, 113, 5, 138, 101, 47, 225, 36, 15, 33,
	53, 147, 142, 218, 240, 18, 130, 69, 29, 181, 194, 125, 106, 39, 249, 185, 201, 154, 9, 120,
	77, 228, 114, 166, 6, 191, 139, 98, 102, 221, 48, 253, 226, 152, 37, 179, 16, 145, 34, 136,
	54, 208, 148, 206, 143, 150, 219, 189, 241, 210, 19, 92, 131, 56, 70, 64, 30, 66, 182, 163,
	195, 72, 126, 110, 107, 58, 40, 84, 250, 133, 186, 61, 202, 94, 155, 159, 10, 21, 121, 43,
	78, 212, 229, 172, 115, 243, 167, 87, 7, 112, 192, 247, 140, 128, 99, 13, 103, 74, 222, 237,
	49, 197, 254, 24, 227, 165, 153, 119, 38, 184, 180, 124, 17, 68, 146, 217, 35, 32, 137, 46,
	55, 63, 209, 91, 149, 188, 207, 205, 144, 135, 151, 178, 220, 252, 190, 97, 242, 86, 211, 171,
	20, 42, 93, 158, 132, 60, 57, 83, 71, 109, 65, 162, 31, 45, 67, 216, 183, 123, 164, 118,
	196, 23, 73, 236, 127, 12, 111, 246, 108, 161, 59, 82, 41, 157, 85, 170, 251, 96, 134, 177,
	187, 204, 62, 90, 203, 89, 95, 176, 156, 169, 160, 81, 11, 245, 22, 235, 122, 117, 44, 215,
	79, 174, 213, 233, 230, 231, 173, 232, 116, 214, 244, 234, 168, 80, 88, 175,
}

// Generator polynomials keyed by ECC word count
var qrGenPolys = map[int][]int{
	7:  {127, 122, 154, 164, 11, 68, 117},
	10: {216, 194, 159, 111, 199, 94, 95, 113, 157, 193},
	13: {137, 73, 227, 17, 177, 17, 52, 13, 46, 43, 83, 132, 120},
	15: {29, 196, 111, 163, 112, 74, 10, 105, 105, 139, 132, 151, 32, 134, 26},
	16: {59, 13, 104, 189, 68, 209, 30, 8, 163, 65, 41, 229, 98, 50, 36, 59},
	17: {119, 66, 83, 120, 119, 22, 197, 83, 249, 41, 143, 134, 85, 53, 125, 99, 79},
	18: {239, 251, 183, 113, 149, 175, 199, 215, 240, 220, 73, 82, 173, 75, 32, 67, 217, 146},
	20: {152, 185, 240, 5, 111, 99, 6, 220, 112, 150, 69, 36, 187, 22, 228, 198, 121, 121, 165, 174},
	22: {89, 179, 131, 176, 182, 244, 19, 189, 69, 40, 28, 137, 29, 123, 67, 253, 86, 218, 230, 26, 145, 245},
	24: {122, 118, 169, 70, 178, 237, 216, 102, 115, 150, 229, 73, 130, 72, 61, 43, 206, 1, 237, 247, 127, 217, 144, 117},
	26: {246, 51, 183, 4, 136, 98, 199, 152, 77, 56, 206, 24, 145, 40, 209, 117, 233, 42, 135, 68, 70, 144, 146, 77, 43, 94},
	28: {252, 9, 28, 13, 18, 251, 208, 150, 103, 174, 100, 41, 167, 12, 247, 56, 117, 119, 233, 127, 181, 100, 121, 147, 176, 74, 58, 197},
	30: {212, 246, 77, 73, 195, 192, 75, 98, 5, 70, 103, 177, 22, 217, 138, 51, 181, 246, 72, 25, 18, 46, 228, 74, 216, 195, 11, 106, 130, 150},
}

// Capacity tables
var qrArrayCap = [41]int{
	0, 26, 44, 70, 100, 134, 172, 196, 242, 292,
	346, 404, 466, 532, 581, 655, 733, 815, 901, 991,
	1085, 1156, 1258, 1364, 1474, 1588, 1706, 1828, 1921, 2051,
	2185, 2323, 2465, 2611, 2761, 2876, 3034, 3196, 3362, 3532,
	3706,
}

var qrArray2Extra = [41]int{
	0, 0, 7, 7, 7, 7, 7, 0, 0, 0,
	0, 0, 0, 0, 3, 3, 3, 3, 3, 3,
	3, 4, 4, 4, 4, 4, 4, 4, 3, 3,
	3, 3, 3, 3, 3, 0, 0, 0, 0, 0,
	0,
}

// qrArray3Cap[4][41] - bit capacity by ECC level and version
var qrArray3Cap = [4][41]int{
	{0, 152, 272, 440, 640, 864, 1088, 1248, 1552, 1856,
		2192, 2592, 2960, 3424, 3688, 4184, 4712, 5176, 5768, 6360,
		6888, 7456, 8048, 8752, 9392, 10208, 10960, 11744, 12248, 13048,
		13880, 14744, 15640, 16568, 17528, 18448, 19472, 20528, 21616, 22496,
		23648},
	{0, 128, 224, 352, 512, 688, 864, 992, 1232, 1456,
		1728, 2032, 2320, 2672, 2920, 3320, 3624, 4056, 4504, 5016,
		5352, 5712, 6256, 6880, 7312, 8000, 8496, 9024, 9544, 10136,
		10984, 11640, 12328, 13048, 13800, 14496, 15312, 15936, 16816, 17728,
		18672},
	{0, 104, 176, 272, 384, 496, 608, 704, 880, 1056,
		1232, 1440, 1648, 1952, 2088, 2360, 2600, 2936, 3176, 3560,
		3880, 4096, 4544, 4912, 5312, 5744, 6032, 6464, 6968, 7288,
		7880, 8264, 8920, 9368, 9848, 10288, 10832, 11408, 12016, 12656,
		13328},
	{0, 72, 128, 208, 288, 368, 480, 528, 688, 800,
		976, 1120, 1264, 1440, 1576, 1784, 2024, 2264, 2504, 2728,
		3080, 3248, 3536, 3712, 4112, 4304, 4768, 5024, 5288, 5608,
		5960, 6344, 6760, 7208, 7688, 7888, 8432, 8768, 9136, 9776,
		10208},
}

// Format information strings (32 patterns)
var qrFormatInfo = [32]string{
	"111011111000100", "111001011110011", "111110110101010", "111100010011101",
	"110011000101111", "110001100011000", "110110001000001", "110100101110110",
	"101010000010010", "101000100100101", "101111001111100", "101101101001011",
	"100010111111001", "100000011001110", "100111110010111", "100101010100000",
	"011010101011111", "011000001101000", "011111100110001", "011101000000110",
	"010010010110100", "010000110000011", "010111011011010", "010101111101101",
	"001011010001001", "001001110111110", "001110011100111", "001100111010000",
	"000011101100010", "000001001010101", "000110100001100", "000100000111011",
}

// ---------------------------------------------------------------------------
// Internal methods
// ---------------------------------------------------------------------------

func qrGet1DAddress(px, py, symbolSize int) int {
	return px + py*(symbolSize+1)
}

func (j *qrJam) qrCodeGetFormatX(pVersion int) [15]int {
	var arr [15]int
	for i := 0; i < 7; i++ {
		arr[i] = 8
	}
	for i := 0; i < 8; i++ {
		arr[i+7] = 4*pVersion + 9 + i
	}
	return arr
}

func (j *qrJam) qrCodeGetFormatY(pVersion int) [15]int {
	var arr [15]int
	for i := 0; i < 8; i++ {
		arr[i+7] = 8
	}
	for i := 0; i < 7; i++ {
		arr[i] = 4*pVersion + 16 - i
	}
	return arr
}

func (j *qrJam) qrCodeInquireBits(data []byte) {
	length := len(data)

	if j.encodeMode == "N" {
		rem := length % 3
		if rem == 0 {
			j.lQRInquireNumericBits = int(math.Round(math.Floor(float64(length)/3.0) * 10.0))
		} else if rem == 1 {
			j.lQRInquireNumericBits = int(math.Round(math.Floor(float64(length)/3.0)*10.0 + 4.0))
		} else if rem == 2 {
			j.lQRInquireNumericBits = int(math.Round(math.Floor(float64(length)/3.0)*10.0 + 7.0))
		}
		j.lQRInquireAlphaNumericBits = 0
		j.lQRInquireKanjiBits = 0
	} else if j.encodeMode == "A" {
		rem := length % 2
		if rem == 0 {
			j.lQRInquireAlphaNumericBits = int(math.Round(math.Floor(float64(length)/2.0) * 11.0))
		} else if rem == 1 {
			j.lQRInquireAlphaNumericBits = int(math.Round(math.Floor(float64(length)/2.0)*11.0 + 6.0))
		}
		j.lQRInquireNumericBits = 0
		j.lQRInquireKanjiBits = 0
	} else {
		j.lQRInquireNumericBits = 0
		j.lQRInquireAlphaNumericBits = 0
		j.lQRInquireKanjiBits = 0
	}
	j.lQRInquireBinaryBits = length * 8
}

func (j *qrJam) qrCodeSetInquireTextType(data []byte) int {
	if j.encodeMode == "N" {
		return qrTextNumeric
	} else if j.encodeMode == "A" {
		return qrTextAlphaNumeric
	}
	return qrTextBinary
}

func (j *qrJam) qrCodeGetSymbolData(pVersion int) []int {
	num := pVersion*4 + 17
	array3 := make([]int, num*(num+1))
	array := [2]int{3, num - 4}

	ap := qrAlignmentPatterns[pVersion]
	if ap == nil {
		ap = []int{0}
	}

	ga := qrGet1DAddress

	for jj := 0; jj < len(array); jj++ {
		for kk := 0; kk < len(array); kk++ {
			num8 := array[jj]
			num9 := array[kk]
			if jj == len(array)-1 && kk == len(array)-1 {
				continue
			}

			// Center and immediate neighbors
			for dx := -1; dx <= 1; dx++ {
				for dy := -1; dy <= 1; dy++ {
					array3[ga(num8+dx, num9+dy, num)] = 3
				}
			}

			// Outer ring
			for i := -3; i <= 3; i++ {
				array3[ga(num8+i, num9-3, num)] = 3
				array3[ga(num8+i, num9+3, num)] = 3
			}
			for i := -2; i <= 2; i++ {
				array3[ga(num8-3, num9+i, num)] = 3
				array3[ga(num8+3, num9+i, num)] = 3
			}

			// Inner ring
			for i := -2; i <= 2; i++ {
				array3[ga(num8+i, num9-2, num)] = 2
				array3[ga(num8+i, num9+2, num)] = 2
			}
			for i := -1; i <= 1; i++ {
				array3[ga(num8-2, num9+i, num)] = 2
				array3[ga(num8+2, num9+i, num)] = 2
			}

			// Border markings for specific corners
			if jj == 0 && kk == 0 {
				for i := -3; i <= 4; i++ {
					array3[ga(num8+4, num9+i, num)] = 8
				}
				for i := -3; i <= 4; i++ {
					array3[ga(num8+i, num9+4, num)] = 8
				}
			}

			if jj == 0 && kk == 1 {
				for i := -4; i <= 3; i++ {
					array3[ga(num8+4, num9+i, num)] = 8
				}
				for i := -4; i <= 3; i++ {
					array3[ga(num8-i, num9-4, num)] = 8
				}
			}

			if jj == 1 && kk == 0 {
				for i := -3; i <= 4; i++ {
					array3[ga(num8-4, num9+i, num)] = 8
				}
				for i := -4; i <= 3; i++ {
					array3[ga(num8+i, num9+4, num)] = 8
				}
			}
		}
	}

	// Alignment patterns
	num16 := len(ap) - 1
	for lIdx := 0; lIdx <= num16; lIdx++ {
		for mIdx := 0; mIdx < len(ap); mIdx++ {
			num18 := ap[lIdx]
			num19 := ap[mIdx]
			if (lIdx == 0 && mIdx == 0) ||
				(lIdx == 0 && mIdx == len(ap)-1) ||
				(mIdx == 0 && lIdx == len(ap)-1) {
				continue
			}

			array3[ga(num18, num19, num)] = 5
			// Outer ring
			for i := -2; i <= 2; i++ {
				array3[ga(num18+i, num19-2, num)] = 5
				array3[ga(num18+i, num19+2, num)] = 5
			}
			for i := -1; i <= 1; i++ {
				array3[ga(num18-2, num19+i, num)] = 5
				array3[ga(num18+2, num19+i, num)] = 5
			}
			// Inner ring
			for dx := -1; dx <= 1; dx++ {
				for dy := -1; dy <= 1; dy++ {
					if dx == 0 && dy == 0 {
						continue
					}
					array3[ga(num18+dx, num19+dy, num)] = 4
				}
			}
			array3[ga(num18, num19, num)] = 5 // center
		}
	}

	// Timing patterns
	num20 := num - 9
	for n := 8; n <= num20; n += 2 {
		array3[ga(n, 6, num)] |= 0x11
		array3[ga(6, n, num)] |= 0x11
		if n != num20 {
			array3[ga(n+1, 6, num)] |= 0x10
			array3[ga(6, n+1, num)] |= 0x10
		}
	}

	array3[ga(8, num-8, num)] = 129

	// Version information
	if pVersion >= 7 {
		num21 := qrVersionInfo[pVersion-7]
		for num22 := 0; num22 < 6; num22++ {
			for num23 := 0; num23 < 3; num23++ {
				num24 := num21 & 1
				if num24 == 1 {
					array3[ga(num-11+num23, num22, num)] |= 0x21
					array3[ga(num22, num-11+num23, num)] |= 0x21
				} else {
					array3[ga(num-11+num23, num22, num)] |= 0x20
					array3[ga(num22, num-11+num23, num)] |= 0x20
				}
				num21 >>= 1
			}
		}
	}

	fmtX := j.qrCodeGetFormatX(j.lQRInquireVersion)
	fmtY := j.qrCodeGetFormatY(j.lQRInquireVersion)
	j.lFormatSecondX = fmtX
	j.lFormatSecondY = fmtY
	for num25 := 0; num25 < 15; num25++ {
		array3[ga(j.lFormatFirstX[num25], j.lFormatFirstY[num25], num)] |= 0x40
		array3[ga(j.lFormatSecondX[num25], j.lFormatSecondY[num25], num)] |= 0x40
	}

	return array3
}

func (j *qrJam) qrCodeTransRSBlock(pDeployData []int) []int {
	eccIdx := j.lQRSetECCRate
	verIdx := j.lQRInquireVersion - 1
	num := gDataWords[eccIdx][verIdx]
	num2 := gExtraDataWords[eccIdx][verIdx]
	num3 := float64(gECCWords[eccIdx][verIdx])
	num4 := gRSBlockCount[eccIdx][verIdx]

	arr := make([]int, len(pDeployData))
	num5 := 0
	i := 0
	array2 := make([]int, 129)
	num6 := 0

	for jj := 0; jj < num; jj++ {
		for kk := 0; kk < num4; kk++ {
			num9 := 0
			if kk > num4-num2 {
				num9 = kk - (num4 - num2)
				if jj == 0 {
					array2[num6] = jj + kk*num + num9 - 1
					num6++
				}
			}
			for num10 := 0; num10 < 8; num10++ {
				num5 = (jj + kk*num + num9) * 8 + num10
				arr[num5] = pDeployData[i]
				i++
			}
		}
	}

	if num2 > 0 {
		array2[num6] = num*num4 + num2 - 1
		num6++
	}

	for lIdx := 0; lIdx < num6; lIdx++ {
		for num12 := 0; num12 < 8; num12++ {
			num5 = array2[lIdx]*8 + num12
			arr[num5] = pDeployData[i]
			i++
		}
	}

	num13 := i
	num14 := int(math.Round(num3 - 1.0))
	for m := 0; m <= num14; m++ {
		for n := 0; n < num4; n++ {
			for num16 := 0; num16 < 8; num16++ {
				num5 = int(math.Round(float64(num13) + (float64(m)+float64(n)*num3)*8.0 + float64(num16)))
				arr[num5] = pDeployData[i]
				i++
			}
		}
	}

	for i < len(pDeployData) {
		arr[i] = pDeployData[i]
		i++
	}

	return arr
}

func qrGetAlphaNumericCode(c byte) int {
	if c >= 0x30 && c <= 0x39 {
		return int(c) - 0x30
	}
	if c >= 0x41 && c <= 0x5A {
		return int(c) - 0x41 + 10
	}
	mapping := map[byte]int{
		0x20: 36, 0x24: 37, 0x25: 38, 0x2A: 39, 0x2B: 40,
		0x2D: 41, 0x2E: 42, 0x2F: 43, 0x3A: 44,
	}
	if v, ok := mapping[c]; ok {
		return v
	}
	return 0
}

func (j *qrJam) qrCodeGet8BitData(pInputData []int, pInputDataBits []int, pMaxBytes int) []int {
	arr := make([]int, pMaxBytes)
	numItems := len(pInputDataBits)
	num2 := 0
	num3 := 8
	num4 := 0

	for i := 0; i < numItems; i++ {
		num4 += pInputDataBits[i]
	}

	num6 := int(math.Floor(float64(num4-1) / 8.0))
	for jj := 0; jj <= num6; jj++ {
		arr[jj] = 0
	}

	for k := 0; k < numItems; k++ {
		num8 := pInputData[k]
		num9 := pInputDataBits[k]
		if num9 == 0 {
			break
		}

		flag := true
		for flag {
			if num3 > num9 {
				arr[num2] = ((arr[num2] << (num9 & 7)) | num8) & 0xFF
				num3 -= num9
				flag = false
			} else {
				num9 -= num3
				arr[num2] = ((arr[num2] << (num3 & 7)) | (num8 >> num9)) & 0xFF
				if num9 == 0 {
					flag = false
				} else {
					num8 &= (1 << num9) - 1
					flag = true
				}
				num2++
				num3 = 8
			}
		}
	}

	if num3 != 8 {
		arr[num2] = (arr[num2] << (num3 & 7)) & 0xFF
	} else {
		num2--
	}

	if num2 < pMaxBytes-1 {
		flag := true
		for num2 < pMaxBytes-1 {
			num2++
			if flag {
				arr[num2] = 236
			} else {
				arr[num2] = 17
			}
			flag = !flag
		}
	}

	return arr
}

func qrCodeGetByteData(pInput1 []int, pInput2 []int) []int {
	var arr, arr2 []int
	if len(pInput1) > len(pInput2) {
		arr = make([]int, len(pInput1))
		copy(arr, pInput1)
		arr2 = make([]int, len(pInput2))
		copy(arr2, pInput2)
	} else {
		arr = make([]int, len(pInput2))
		copy(arr, pInput2)
		arr2 = make([]int, len(pInput1))
		copy(arr2, pInput1)
	}

	numA := len(arr)
	num2 := len(arr2)
	arr3 := make([]int, numA+1)

	for i := 0; i < numA; i++ {
		if i < num2 {
			arr3[i] = arr[i] ^ arr2[i]
		} else {
			arr3[i] = arr[i]
		}
	}

	return arr3
}

func (j *qrJam) qrCodeGetTotalData(pInputData []int, pECC int, pData int, pTotal int) []int {
	eccIdx := j.lQRSetECCRate
	verIdx := j.lQRInquireVersion - 1
	num := gDataWords[eccIdx][verIdx]
	num2 := gExtraDataWords[eccIdx][verIdx]
	num3 := float64(gECCWords[eccIdx][verIdx])
	num4 := gRSBlockCount[eccIdx][verIdx]

	array4 := qrGenPolys[pECC]
	if array4 == nil {
		array4 = []int{}
	}

	// Build multiplication table
	arrTable := make([][]int, 256)
	for i := 0; i < 256; i++ {
		arrTable[i] = make([]int, len(array4))
	}
	for i := 0; i < len(array4); i++ {
		num6Val := gfLog[array4[i]]
		for num7 := 0; num7 < 256; num7++ {
			num8 := (num6Val + gfLog[num7]) % 255
			num9 := gfExp[num8]
			arrTable[num7][i] = num9
		}
	}

	// RS block sizes
	j.lRSBlock = make([]int, num4)
	for jj := 0; jj < num4; jj++ {
		if jj >= num4-num2 {
			j.lRSBlock[jj] = int(math.Round(float64(num) + num3 + 1.0))
		} else {
			j.lRSBlock[jj] = int(math.Round(float64(num) + num3))
		}
	}

	// Split input data into RS blocks
	array5 := make([][]int, len(j.lRSBlock))
	array6 := make([]int, pTotal)
	copy(array6, pInputData)

	for num10Idx := 0; num10Idx < len(j.lRSBlock); num10Idx++ {
		for k := 0; k < len(j.lRSBlock); k++ {
			size := j.lRSBlock[num10Idx] - pECC
			array5[k] = make([]int, size+1)
		}
	}

	num11 := 0
	num12 := 0
	for num10 := 0; num10 < pData; num10++ {
		array5[num12][num11] = pInputData[num10]
		num11++
		if num11 >= j.lRSBlock[num12]-pECC {
			num11 = 0
			num12++
		}
	}

	for num12Idx := 0; num12Idx < len(j.lRSBlock); num12Idx++ {
		array8 := make([]int, len(array5[num12Idx]))
		copy(array8, array5[num12Idx])
		num15 := j.lRSBlock[num12Idx]
		num16 := num15 - pECC

		for cnt := num16; cnt > 0; cnt-- {
			b := array8[0]
			if b != 0 {
				array9 := make([]int, len(array8)-1)
				copy(array9, array8[1:])
				array10 := make([]int, pECC)
				for lIdx := 0; lIdx < pECC; lIdx++ {
					array10[lIdx] = arrTable[b][lIdx]
				}
				array8 = qrCodeGetByteData(array9, array10)
			} else if pECC < len(array8) {
				array8 = array8[1:]
			} else {
				array12 := make([]int, pECC)
				for idx := 0; idx < pECC && idx < len(array8)-1; idx++ {
					array12[idx] = array8[idx+1]
				}
				array12[pECC-1] = 0
				array8 = array12
			}
		}

		destStart := len(pInputData) + num12Idx*pECC
		for idx := 0; idx < pECC; idx++ {
			if destStart+idx < len(array6) && idx < len(array8) {
				array6[destStart+idx] = array8[idx]
			}
		}
	}

	return array6
}

func qrCodeGetMaskBit(maskNo, py, px int) bool {
	switch maskNo {
	case 0:
		return (px+py)%2 == 0
	case 1:
		return px%2 == 0
	case 2:
		return py%3 == 0
	case 3:
		return (px+py)%3 == 0
	case 4:
		return (px/2+py/3)%2 == 0
	case 5:
		return (px*py)%2+(px*py)%3 == 0
	case 6:
		return ((px*py)%2+(px*py)%3)%2 == 0
	case 7:
		return ((px*py)%3+(px+py)%2)%2 == 0
	}
	return false
}

func qrCodeGetMaskBitGroup(py, px int) int {
	num := 0
	for m := 0; m < 8; m++ {
		if qrCodeGetMaskBit(m, py, px) {
			num += 1 << m
		}
	}
	return num
}

func qrCodeChooseMaskNumber(pModuleData [][]int, pExtraBits int) int {
	length := len(pModuleData)
	array := make([]int, 8)
	array2 := make([]int, 8)
	array3 := make([]int, 8)
	array4 := make([]int, 8)
	array5 := make([]int, 8)
	array6 := make([]int, 8)
	array7 := make([]int, 9)
	array8 := make([]int, 9)
	array9 := make([]bool, 9)
	array10 := make([]bool, 9)

	for i := 0; i < length; i++ {
		for n := 0; n < 8; n++ {
			array7[n] = 0
			array8[n] = 0
			array9[n] = false
			array10[n] = false
		}

		for jj := 0; jj < length; jj++ {
			num := 0
			num2 := 0
			if jj > 0 && i > 0 {
				num = pModuleData[jj][i] & pModuleData[jj-1][i] &
					pModuleData[jj][i-1] & pModuleData[jj-1][i-1]
				num2 = pModuleData[jj][i] | pModuleData[jj-1][i] |
					pModuleData[jj][i-1] | pModuleData[jj-1][i-1]
			}

			for m := 0; m < 8; m++ {
				array7[m] = ((array7[m] & 0x3F) << 1) | ((pModuleData[jj][i] >> (m & 7)) & 1)
				array8[m] = ((array8[m] & 0x3F) << 1) | ((pModuleData[i][jj] >> (m & 7)) & 1)

				if pModuleData[jj][i]&(1<<m) != 0 {
					array4[m]++
				}
				if array7[m] == 0b1011101 {
					array3[m] += 40
				}
				if array8[m] == 0b1011101 {
					array3[m] += 40
				}
				if jj > 0 && i > 0 {
					if (num&1) != 0 || (num2&1) == 0 {
						array2[m] += 3
					}
					num >>= 1
					num2 >>= 1
				}

				if (array7[m]&0x1F) == 0 || (array7[m]&0x1F) == 31 {
					if jj > 3 {
						if array9[m] {
							array[m]++
						} else {
							array[m] += 3
							array9[m] = true
						}
					}
				} else {
					array9[m] = false
				}

				if (array8[m]&0x1F) == 0 || (array8[m]&0x1F) == 31 {
					if jj > 3 {
						if array10[m] {
							array[m]++
						} else {
							array[m] += 3
							array10[m] = true
						}
					}
				} else {
					array10[m] = false
				}
			}
		}
	}

	num8 := 0
	result := 0
	array11 := [21]int{90, 80, 70, 60, 50, 40, 30, 20, 10, 0, 0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 90}

	for n := 0; n < 8; n++ {
		idx := int(math.Round(float64(int((20.0 * float64(array4[n])) / float64(pExtraBits)))))
		if idx < 0 {
			idx = 0
		}
		if idx > 20 {
			idx = 20
		}
		array5[n] = array11[idx]
		array6[n] = array[n] + array2[n] + array3[n] + array5[n]
	}

	for o := 0; o < 8; o++ {
		if array6[o] < num8 || o == 0 {
			result = o
			num8 = array6[o]
		}
	}

	return result
}

func (j *qrJam) qrGetData(data []byte) ([][]int, error) {
	num4 := len(data)
	j.qrCodeInquireBits(data)
	j.lQRInquireTextType = j.qrCodeSetInquireTextType(data)
	qrTextTypes := j.lQRInquireTextType

	if qrTextTypes == qrTextKanji {
		qrTextTypes = qrTextBinary
		j.lQRInquireTextType = qrTextBinary
	}

	array6 := make([]int, num4+32)
	array7 := make([]int, num4+32)

	if num4 <= 0 {
		return [][]int{{0}}, nil
	}

	array7[0] = 4
	num := 0 // index into array6/array7

	// Version-dependent extra bits for character count indicator
	var array5Ver [41]int

	if qrTextTypes == qrTextNumeric {
		array5Ver = [41]int{
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
			2, 2, 2, 2, 2, 2, 2, 4, 4, 4,
			4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
		}
		array6[num] = 1
		num++
		array6[num] = num4
		array7[num] = 10
		num++

		for jj := 0; jj < num4; jj++ {
			b := int(data[jj]) - 0x30
			if jj%3 == 0 {
				array6[num] = b
				array7[num] = 4
			} else {
				array6[num] = array6[num]*10 + b
				if jj%3 == 1 {
					array7[num] = 7
				} else {
					array7[num] = 10
					if jj < num4-1 {
						num++
					}
				}
			}
		}
		num++
	} else {
		// Binary / default
		array5Ver = [41]int{
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
			8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
			8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
		}
		array6[num] = 4 // Binary mode indicator
		num++
		array6[num] = num4
		array7[num] = 8
		num++

		for i := 0; i < num4; i++ {
			array6[i+num] = int(data[i]) & 0xFF
			array7[i+num] = 8
		}
		num += num4
	}

	// Calculate total bits
	num2 := 0
	for lIdx := 0; lIdx < num; lIdx++ {
		num2 += array7[lIdx]
	}

	// num5 is the index of the character count indicator
	// For numeric mode, it's at index 1; for binary, also at index 1
	num5 := 1

	// Auto version selection
	j.lQRInquireVersion = 1
	num12 := 1
	num3 := 0
	for num12 <= 40 {
		if qrArray3Cap[j.lQRSetECCRate][num12] >= num2+array5Ver[j.lQRInquireVersion] {
			num3 = qrArray3Cap[j.lQRSetECCRate][num12]
			break
		}
		j.lQRInquireVersion++
		num12++
	}

	if j.lQRInquireVersion > 40 {
		return nil, fmt.Errorf("data too long for QR Code")
	}

	// Version adjustment (C# QR.cs compatible logic)
	if j.lQRInquireVersion != 0 {
		if j.lQRInquireVersion > j.lQRSetVersion {
			num3 = qrArray3Cap[j.lQRSetECCRate][j.lQRInquireVersion]
		} else {
			j.lQRInquireVersion = j.lQRSetVersion
			num3 = qrArray3Cap[j.lQRSetECCRate][j.lQRInquireVersion]
		}
	}

	num2 += array5Ver[j.lQRInquireVersion]
	array7[num5] = array7[num5] + array5Ver[j.lQRInquireVersion]
	num14 := qrArrayCap[j.lQRInquireVersion]
	num15 := qrArray2Extra[j.lQRInquireVersion] + (num14 << 3)

	fmtX := j.qrCodeGetFormatX(j.lQRInquireVersion)
	fmtY := j.qrCodeGetFormatY(j.lQRInquireVersion)
	j.lFormatSecondX = fmtX
	j.lFormatSecondY = fmtY

	pECC := gECCWords[j.lQRSetECCRate][j.lQRInquireVersion-1]
	num16 := int(math.Round(float64(num3) / 8.0))
	num17 := 4*j.lQRInquireVersion + 17
	j.lQRInquireModuleSize = num17

	array9 := j.qrCodeGetSymbolData(j.lQRInquireVersion)

	j.lDeployTableX = make([]int, num15+1)
	j.lDeployTableY = make([]int, num15+1)
	j.lDeployOrderX = make([]int, num15)
	j.lDeployOrderY = make([]int, num15)

	// Data placement order
	num19 := 0
	mVal := num17 - 1
	for mVal >= 3 {
		for nVal := num17 - 1; nVal >= 0; nVal-- {
			num20 := nVal*(num17+1) + mVal
			if (array9[num20] & 0xFE) == 0 {
				j.lDeployTableX[num19] = mVal
				j.lDeployTableY[num19] = nVal
				num19++
			}
			num20 = nVal*(num17+1) + mVal - 1
			if (array9[num20] & 0xFE) == 0 {
				j.lDeployTableX[num19] = mVal - 1
				j.lDeployTableY[num19] = nVal
				num19++
			}
		}

		if mVal == 8 {
			mVal = 7
		}
		for num22 := 0; num22 < num17; num22++ {
			num23 := num22*(num17+1) + mVal - 2
			if (array9[num23] & 0xFE) == 0 {
				j.lDeployTableX[num19] = mVal - 2
				j.lDeployTableY[num19] = num22
				num19++
			}
			num23 = num22*(num17+1) + mVal - 3
			if (array9[num23] & 0xFE) == 0 {
				j.lDeployTableX[num19] = mVal - 3
				j.lDeployTableY[num19] = num22
				num19++
			}
		}
		mVal -= 4
	}

	j.lDeployOrderX = j.qrCodeTransRSBlock(j.lDeployTableX)
	j.lDeployOrderY = j.qrCodeTransRSBlock(j.lDeployTableY)

	// Terminator
	if num2 <= num3-4 {
		array6[num] = 0
		array7[num] = 4
	} else if num2 < num3 {
		array6[num] = 0
		array7[num] = num3 - num2
	}

	// Convert to 8-bit data
	pInputData := j.qrCodeGet8BitData(array6, array7, num16)

	// Add error correction data
	array10 := j.qrCodeGetTotalData(pInputData, pECC, num16, num14)

	// Initialize QR matrix
	array11 := make([][]int, num17)
	for i := range array11 {
		array11[i] = make([]int, num17)
	}

	// Place data into matrix
	for num29 := 0; num29 < num14; num29++ {
		b5 := array10[num29]
		for num30 := 7; num30 >= 0; num30-- {
			num31 := num29*8 + num30
			num32 := qrCodeGetMaskBitGroup(
				j.lDeployOrderX[num31], j.lDeployOrderY[num31])
			array11[j.lDeployOrderX[num31]][j.lDeployOrderY[num31]] =
				(255 * (b5 & 1)) ^ num32
			b5 >>= 1
		}
	}

	for num33 := qrArray2Extra[j.lQRInquireVersion]; num33 > 0; num33-- {
		num34 := num33 + num14*8 - 1
		num32 := qrCodeGetMaskBitGroup(
			j.lDeployOrderX[num34], j.lDeployOrderY[num34])
		array11[j.lDeployOrderX[num34]][j.lDeployOrderY[num34]] = 0xFF ^ num32
	}

	// Choose mask pattern
	b6 := qrCodeChooseMaskNumber(
		array11, qrArray2Extra[j.lQRInquireVersion]+num14*8)
	b7 := int(math.Round(math.Pow(2.0, float64(b6))))
	b8 := b6 + j.lQRSetECCRate*8

	// Place format information
	for num35 := 0; num35 < 15; num35++ {
		b9 := int(qrFormatInfo[b8][num35] - '0')
		array11[j.lFormatFirstX[num35]][j.lFormatFirstY[num35]] = b9 * 255
		array11[j.lFormatSecondX[num35]][j.lFormatSecondY[num35]] = b9 * 255
	}

	// Build final matrix
	array12 := make([][]int, num17)
	for i := range array12 {
		array12[i] = make([]int, num17)
	}
	num36 := 0
	for num38 := 0; num38 < num17; num38++ {
		for num40 := 0; num40 < num17; num40++ {
			if (array11[num40][num38]&b7) != 0 || (array9[num36]&1) == 1 {
				array12[num40][num38] = 1 | (array9[num36] & 0xFE)
			} else {
				array12[num40][num38] = 0 | (array9[num36] & 0xFE)
			}
			num36++
		}
		num36++
	}

	return array12, nil
}

func (j *qrJam) makeArr(data []byte) error {
	if len(data) == 0 {
		return fmt.Errorf("empty data")
	}

	array, err := j.qrGetData(data)
	if err != nil {
		return err
	}
	size := len(array)
	if size <= 0 {
		return fmt.Errorf("QR encoding produced empty array")
	}

	j.patt = make([][]bool, size)
	for i := 0; i < size; i++ {
		j.patt[i] = make([]bool, size)
		for jj := 0; jj < size; jj++ {
			b := array[i][jj]
			j.patt[i][jj] = (b & 1) == 1
		}
	}

	return nil
}

// Ensure QRCode satisfies the interface expected by fillRect, etc.
var _ color.Color = RGBA{}
