package barcode

import (
	"strings"
	"testing"
)

func TestPDF417EncodeBasic(t *testing.T) {
	enc := newPDF417Encoder()
	enc.text = []byte("Hello PDF417")
	enc.useAutoErrorLevel = true
	enc.aspectRatio = 0.5
	enc.yHeight = 3

	err := enc.paintCode()
	if err != nil {
		t.Fatalf("paintCode failed: %v", err)
	}
	if enc.codeRows == 0 {
		t.Fatal("codeRows is 0")
	}
	if enc.codeColumns == 0 {
		t.Fatal("codeColumns is 0")
	}
	if len(enc.outBits) == 0 {
		t.Fatal("outBits is empty")
	}

	matrix := enc.convertToBoolMatrix()
	if matrix == nil {
		t.Fatal("convertToBoolMatrix returned nil")
	}
	if len(matrix) != enc.codeRows {
		t.Errorf("expected %d rows, got %d", enc.codeRows, len(matrix))
	}
	if len(matrix[0]) != enc.bitColumns {
		t.Errorf("expected %d columns, got %d", enc.bitColumns, len(matrix[0]))
	}

	// Verify at least some modules are black
	hasBlack := false
	for _, row := range matrix {
		for _, v := range row {
			if v {
				hasBlack = true
				break
			}
		}
		if hasBlack {
			break
		}
	}
	if !hasBlack {
		t.Error("matrix has no black modules")
	}
}

func TestPDF417DrawSVG(t *testing.T) {
	p := NewPDF417(FormatSVG)
	err := p.Draw("Hello PDF417", 400, 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := p.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
	if !strings.Contains(svg, "</svg>") {
		t.Error("SVG output missing </svg> tag")
	}
	if !strings.Contains(svg, "<rect") {
		t.Error("SVG output missing <rect elements")
	}
}

func TestPDF417DrawPNG(t *testing.T) {
	p := NewPDF417(FormatPNG)
	err := p.Draw("Hello PDF417", 400, 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := p.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
	// Check PNG magic bytes
	if data[0] != 0x89 || data[1] != 0x50 {
		t.Error("output is not valid PNG")
	}
}

func TestPDF417AutoDimensions(t *testing.T) {
	// With auto sizing and different data lengths, verify dimensions are reasonable
	tests := []struct {
		data    string
		minRows int
		maxRows int
	}{
		{"A", 3, 10},
		{"Hello World", 3, 20},
		{strings.Repeat("A", 100), 5, 90},
		{strings.Repeat("A", 500), 10, 90},
	}

	for _, tc := range tests {
		p := NewPDF417(FormatSVG)
		patt, err := p.GetPattern(tc.data)
		if err != nil {
			t.Fatalf("GetPattern(%q) failed: %v", tc.data[:min(20, len(tc.data))], err)
		}
		numRows := len(patt)
		if numRows < tc.minRows || numRows > tc.maxRows {
			t.Errorf("GetPattern(%q): rows=%d, expected [%d,%d]",
				tc.data[:min(20, len(tc.data))], numRows, tc.minRows, tc.maxRows)
		}
	}
}

func TestPDF417FixedColumns(t *testing.T) {
	p := NewPDF417(FormatSVG)
	p.SetSizeMode(PDF417SizeColumns)
	p.SetColumns(5)
	p.SetUseAutoErrorLevel(true)

	patt, err := p.GetPattern("Testing fixed columns")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if patt == nil {
		t.Fatal("pattern is nil")
	}
	// Pattern should have been generated successfully
	if len(patt) < 3 {
		t.Errorf("expected at least 3 rows, got %d", len(patt))
	}
}

func TestPDF417ErrorLevels(t *testing.T) {
	data := "Error Level Test"

	// Test with different manual error levels
	for _, level := range []int{0, 2, 4, 6, 8} {
		p := NewPDF417(FormatSVG)
		p.SetUseAutoErrorLevel(false)
		p.SetErrorLevel(level)

		patt, err := p.GetPattern(data)
		if err != nil {
			t.Fatalf("GetPattern with error level %d failed: %v", level, err)
		}
		if patt == nil {
			t.Fatalf("pattern is nil for error level %d", level)
		}
	}

	// Test auto error level
	p := NewPDF417(FormatSVG)
	p.SetUseAutoErrorLevel(true)
	patt, err := p.GetPattern(data)
	if err != nil {
		t.Fatalf("GetPattern with auto error level failed: %v", err)
	}
	if patt == nil {
		t.Fatal("pattern is nil for auto error level")
	}
}

func TestPDF417GetPattern(t *testing.T) {
	p := NewPDF417(FormatSVG)
	patt, err := p.GetPattern("PDF417")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if patt == nil {
		t.Fatal("pattern is nil")
	}

	numRows := len(patt)
	if numRows == 0 {
		t.Fatal("pattern has 0 rows")
	}
	numCols := len(patt[0])
	if numCols == 0 {
		t.Fatal("pattern has 0 columns")
	}

	// All rows should have the same number of columns
	for i, row := range patt {
		if len(row) != numCols {
			t.Errorf("row %d has %d columns, expected %d", i, len(row), numCols)
		}
	}

	// First bit of each row should be black (start pattern starts with 1)
	for i, row := range patt {
		if !row[0] {
			t.Errorf("row %d: first module should be black (start pattern)", i)
		}
	}
}

func TestPDF417NumericMode(t *testing.T) {
	// Long numeric string should trigger numeric compaction
	p := NewPDF417(FormatSVG)
	numericData := "1234567890123456789012345678901234567890"
	patt, err := p.GetPattern(numericData)
	if err != nil {
		t.Fatalf("GetPattern for numeric data failed: %v", err)
	}
	if patt == nil {
		t.Fatal("pattern is nil")
	}

	// Compare with text data of similar length - numeric should be more compact
	pText := NewPDF417(FormatSVG)
	textData := "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmn"
	pattText, err := pText.GetPattern(textData)
	if err != nil {
		t.Fatalf("GetPattern for text data failed: %v", err)
	}

	numericModules := len(patt) * len(patt[0])
	textModules := len(pattText) * len(pattText[0])
	// Numeric should produce a similar or smaller barcode
	// (more efficient encoding for digits)
	if numericModules > textModules*2 {
		t.Errorf("numeric encoding (%d modules) is much larger than text encoding (%d modules)",
			numericModules, textModules)
	}
}

func TestPDF417Empty(t *testing.T) {
	p := NewPDF417(FormatSVG)

	// Empty string should produce an error
	_, err := p.GetPattern("")
	if err == nil {
		t.Fatal("expected error for empty string")
	}

	// Draw with empty string should also fail
	err = p.Draw("", 400, 200)
	if err == nil {
		t.Fatal("expected error for empty Draw")
	}
}

func TestPDF417Settings(t *testing.T) {
	p := NewPDF417(FormatPNG)

	// Test defaults
	if p.ErrorLevel() != PDF417ErrorLevel2 {
		t.Errorf("default error level: got %d, want %d", p.ErrorLevel(), PDF417ErrorLevel2)
	}
	if p.Columns() != 0 {
		t.Errorf("default columns: got %d, want 0", p.Columns())
	}
	if p.Rows() != 0 {
		t.Errorf("default rows: got %d, want 0", p.Rows())
	}
	if p.SizeMode() != PDF417SizeAuto {
		t.Errorf("default size mode: got %d, want %d", p.SizeMode(), PDF417SizeAuto)
	}
	if p.AspectRatio() != 0.5 {
		t.Errorf("default aspect ratio: got %f, want 0.5", p.AspectRatio())
	}
	if p.YHeight() != 3 {
		t.Errorf("default y height: got %d, want 3", p.YHeight())
	}
	if !p.UseAutoErrorLevel() {
		t.Error("default UseAutoErrorLevel should be true")
	}

	// Test setters
	p.SetErrorLevel(PDF417ErrorLevel5)
	if p.ErrorLevel() != PDF417ErrorLevel5 {
		t.Errorf("SetErrorLevel: got %d, want %d", p.ErrorLevel(), PDF417ErrorLevel5)
	}
	p.SetColumns(10)
	if p.Columns() != 10 {
		t.Errorf("SetColumns: got %d, want 10", p.Columns())
	}
	p.SetRows(20)
	if p.Rows() != 20 {
		t.Errorf("SetRows: got %d, want 20", p.Rows())
	}
	p.SetSizeMode(PDF417SizeColumnsAndRows)
	if p.SizeMode() != PDF417SizeColumnsAndRows {
		t.Errorf("SetSizeMode: got %d, want %d", p.SizeMode(), PDF417SizeColumnsAndRows)
	}
	p.SetAspectRatio(0.8)
	if p.AspectRatio() != 0.8 {
		t.Errorf("SetAspectRatio: got %f, want 0.8", p.AspectRatio())
	}
	p.SetYHeight(5)
	if p.YHeight() != 5 {
		t.Errorf("SetYHeight: got %d, want 5", p.YHeight())
	}
	p.SetUseAutoErrorLevel(false)
	if p.UseAutoErrorLevel() {
		t.Error("SetUseAutoErrorLevel(false) did not take effect")
	}

	// Test boundary rejection
	p.SetErrorLevel(99)
	if p.ErrorLevel() != PDF417ErrorLevel5 {
		t.Errorf("SetErrorLevel out of range should be ignored, got %d", p.ErrorLevel())
	}
	p.SetColumns(-1)
	if p.Columns() != 10 {
		t.Errorf("SetColumns(-1) should be ignored, got %d", p.Columns())
	}
}

func TestPDF417DrawJPEG(t *testing.T) {
	p := NewPDF417(FormatJPEG)
	err := p.Draw("JPEG Test", 300, 150)
	if err != nil {
		t.Fatalf("Draw JPEG failed: %v", err)
	}
	data := p.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("JPEG output is empty")
	}
	// Check JPEG magic bytes (SOI marker)
	if data[0] != 0xFF || data[1] != 0xD8 {
		t.Error("output is not valid JPEG")
	}
}

func TestPDF417Base64(t *testing.T) {
	p := NewPDF417(FormatPNG)
	err := p.Draw("Base64 Test", 300, 150)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	b64, err := p.GetImageBase64()
	if err != nil {
		t.Fatalf("GetImageBase64 failed: %v", err)
	}
	if !strings.HasPrefix(b64, "data:image/png;base64,") {
		t.Error("base64 output does not have expected prefix")
	}
}

func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}
