package barcode

import (
	"fmt"
	"image"
	"image/draw"
	"math"
)

// PDF417 error level constants.
const (
	PDF417ErrorAuto   = -1
	PDF417ErrorLevel0 = 0
	PDF417ErrorLevel1 = 1
	PDF417ErrorLevel2 = 2
	PDF417ErrorLevel3 = 3
	PDF417ErrorLevel4 = 4
	PDF417ErrorLevel5 = 5
	PDF417ErrorLevel6 = 6
	PDF417ErrorLevel7 = 7
	PDF417ErrorLevel8 = 8
)

// PDF417 size mode constants.
const (
	PDF417SizeAuto          = 0
	PDF417SizeRows          = 1
	PDF417SizeColumns       = 2
	PDF417SizeColumnsAndRows = 3
)

// Internal constants matching the C++/Python implementation.
const (
	pdf417StartPattern       = 0x1fea8
	pdf417StopPattern        = 0x3fa29
	pdf417StartCodeSize      = 17
	pdf417StopSize           = 18
	pdf417Mod                = 929
	pdf417Alpha              = 0x10000
	pdf417Lower              = 0x20000
	pdf417Mixed              = 0x40000
	pdf417Punctuation        = 0x80000
	pdf417IsByte             = 0x100000
	pdf417ByteShift          = 913
	pdf417PL                 = 25
	pdf417LL                 = 27
	pdf417AS                 = 27
	pdf417ML                 = 28
	pdf417AL                 = 28
	pdf417PS                 = 29
	pdf417PAL                = 29
	pdf417Space              = 26
	pdf417TextMode           = 900
	pdf417ByteMode6          = 924
	pdf417ByteMode           = 901
	pdf417NumericMode         = 902
	pdf417AbsoluteMaxTextSize = 5420
	pdf417MaxDataCodewords    = 926
)

// Internal size kind constants.
const (
	pdf417Auto          = 0
	pdf417Columns       = 1
	pdf417Rows          = 2
	pdf417ColumnsAndRows = 3
)

var (
	pdf417MixedSet       = "0123456789&\r\t,:#-.$/+%*=^"
	pdf417PunctuationSet = ";<>@[\\]_`~!\r\t,:\n-.$/\"|*()?{}'"
)

// ---------- Internal segment ----------

type pdf417Segment struct {
	segType byte // 'T', 'N', 'B', or 0
	start   int
	end     int
}

// ---------- Internal encoder ----------

type pdf417Encoder struct {
	outBits           []byte
	bitPtr            int
	bitColumns        int
	errorLevel        int
	lenCodewords      int
	codeColumns       int
	codeRows          int
	useAutoErrorLevel bool
	aspectRatio       float64
	yHeight           int
	options           int
	sizeKind          int
	codewords         []int
	text              []byte
	cwPtr             int
	segmentList       []pdf417Segment
}

func newPDF417Encoder() *pdf417Encoder {
	return &pdf417Encoder{
		errorLevel:        2,
		useAutoErrorLevel: true,
		aspectRatio:       0.5,
		yHeight:           3,
	}
}

func (enc *pdf417Encoder) initBlock() {
	enc.codewords = make([]int, pdf417MaxDataCodewords+2)
}

// ---------- Bit output ----------

func (enc *pdf417Encoder) outCodeword17(codeword int) {
	bytePtr := enc.bitPtr >> 3
	bit := enc.bitPtr - bytePtr*8
	enc.outBits[bytePtr] |= byte((codeword >> (9 + bit)) & 0xFF)
	bytePtr++
	enc.outBits[bytePtr] |= byte((codeword >> (1 + bit)) & 0xFF)
	bytePtr++
	codeword <<= 8
	enc.outBits[bytePtr] |= byte((codeword >> (1 + bit)) & 0xFF)
	enc.bitPtr += 17
}

func (enc *pdf417Encoder) outCodeword18(codeword int) {
	bytePtr := enc.bitPtr >> 3
	bit := enc.bitPtr - bytePtr*8
	enc.outBits[bytePtr] |= byte((codeword >> (10 + bit)) & 0xFF)
	bytePtr++
	enc.outBits[bytePtr] |= byte((codeword >> (2 + bit)) & 0xFF)
	bytePtr++
	codeword <<= 8
	enc.outBits[bytePtr] |= byte((codeword >> (2 + bit)) & 0xFF)
	if bit == 7 {
		bytePtr++
		enc.outBits[bytePtr] |= 0x80
	}
	enc.bitPtr += 18
}

func (enc *pdf417Encoder) outCodeword(codeword int) {
	enc.outCodeword17(codeword)
}

func (enc *pdf417Encoder) outStartPattern() {
	enc.outCodeword17(pdf417StartPattern)
}

func (enc *pdf417Encoder) outStopPattern() {
	enc.outCodeword18(pdf417StopPattern)
}

// ---------- Paint code ----------

func (enc *pdf417Encoder) outPaintCode() {
	enc.bitColumns = pdf417StartCodeSize*(enc.codeColumns+3) + pdf417StopSize
	bytesPerRow := (enc.bitColumns - 1) / 8 + 1
	lenBits := bytesPerRow * enc.codeRows
	enc.outBits = make([]byte, lenBits)

	codePtr := 0
	for row := 0; row < enc.codeRows; row++ {
		enc.bitPtr = bytesPerRow * 8 * row
		rowMod := row % 3
		cluster := &pdf417Clusters[rowMod]

		enc.outStartPattern()

		// Left edge indicator
		var edge int
		switch rowMod {
		case 0:
			edge = 30*(row/3) + (enc.codeRows-1)/3
		case 1:
			edge = 30*(row/3) + enc.errorLevel*3 + (enc.codeRows-1)%3
		default:
			edge = 30*(row/3) + enc.codeColumns - 1
		}
		enc.outCodeword(cluster[edge])

		// Data codewords
		for col := 0; col < enc.codeColumns; col++ {
			if codePtr >= enc.lenCodewords {
				break
			}
			cw := enc.codewords[codePtr]
			if cw < 0 || cw >= len(cluster) {
				break
			}
			enc.outCodeword(cluster[cw])
			codePtr++
		}

		// Right edge indicator
		switch rowMod {
		case 0:
			edge = 30*(row/3) + enc.codeColumns - 1
		case 1:
			edge = 30*(row/3) + (enc.codeRows-1)/3
		default:
			edge = 30*(row/3) + enc.errorLevel*3 + (enc.codeRows-1)%3
		}
		enc.outCodeword(cluster[edge])

		enc.outStopPattern()
	}

	if (enc.options & 0x02) != 0 { // PDF417_INVERT_BITMAP
		for k := range enc.outBits {
			enc.outBits[k] ^= 0xFF
		}
	}
}

// ---------- Error correction ----------

func (enc *pdf417Encoder) calculateErrorCorrection(dest int) {
	el := enc.errorLevel
	if el < 0 || el > 8 {
		el = 0
	}
	A := pdf417ErrorLevel[el]
	ALength := 2 << el
	for k := 0; k < ALength; k++ {
		enc.codewords[dest+k] = 0
	}
	lastE := ALength - 1
	for k := 0; k < enc.lenCodewords; k++ {
		t1 := enc.codewords[k] + enc.codewords[dest]
		for e := 0; e < ALength; e++ {
			t2 := (t1 * A[lastE-e]) % pdf417Mod
			t3 := pdf417Mod - t2
			nxt := 0
			if e != lastE {
				nxt = enc.codewords[dest+e+1]
			}
			enc.codewords[dest+e] = (nxt + t3) % pdf417Mod
		}
	}
	for k := 0; k < ALength; k++ {
		enc.codewords[dest+k] = (pdf417Mod - enc.codewords[dest+k]) % pdf417Mod
	}
}

// ---------- Text type/value ----------

func (enc *pdf417Encoder) getTextTypeAndValue(maxLength, idx int) int {
	if idx >= maxLength {
		return 0
	}
	c := int(enc.text[idx]) & 0xFF
	ch := rune(c)

	if ch >= 'A' && ch <= 'Z' {
		return pdf417Alpha + c - int('A')
	}
	if ch >= 'a' && ch <= 'z' {
		return pdf417Lower + c - int('a')
	}
	if ch == ' ' {
		return pdf417Alpha + pdf417Lower + pdf417Mixed + pdf417Space
	}

	ms := indexOf(pdf417MixedSet, byte(ch))
	ps := indexOf(pdf417PunctuationSet, byte(ch))

	if ms < 0 && ps < 0 {
		return pdf417IsByte + c
	}
	if ms == ps {
		return pdf417Mixed + pdf417Punctuation + ms
	}
	if ms >= 0 {
		return pdf417Mixed + ms
	}
	return pdf417Punctuation + ps
}

// indexOf returns the first index of b in s, or -1.
func indexOf(s string, b byte) int {
	for i := 0; i < len(s); i++ {
		if s[i] == b {
			return i
		}
	}
	return -1
}

// ---------- Text compaction ----------

func (enc *pdf417Encoder) textCompaction(start, length int) error {
	dest := make([]int, pdf417AbsoluteMaxTextSize*2)
	mode := pdf417Alpha
	ptr := 0
	fullBytes := 0
	length += start
	k := start
	for k < length {
		v := enc.getTextTypeAndValue(length, k)
		if (v & mode) != 0 {
			dest[ptr] = v & 0xFF
			ptr++
			k++
			continue
		}
		if (v & pdf417IsByte) != 0 {
			if (ptr & 1) != 0 {
				if (mode & pdf417Punctuation) != 0 {
					dest[ptr] = pdf417PAL
				} else {
					dest[ptr] = pdf417PS
				}
				ptr++
				if (mode & pdf417Punctuation) != 0 {
					mode = pdf417Alpha
				}
			}
			dest[ptr] = pdf417ByteShift
			ptr++
			dest[ptr] = v & 0xFF
			ptr++
			fullBytes += 2
			k++
			continue
		}

		if mode == pdf417Alpha {
			if (v & pdf417Lower) != 0 {
				dest[ptr] = pdf417LL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Lower
			} else if (v & pdf417Mixed) != 0 {
				dest[ptr] = pdf417ML
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Mixed
			} else if (enc.getTextTypeAndValue(length, k+1) &
				enc.getTextTypeAndValue(length, k+2) & pdf417Punctuation) != 0 {
				dest[ptr] = pdf417ML
				ptr++
				dest[ptr] = pdf417PL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Punctuation
			} else {
				dest[ptr] = pdf417PS
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
			}
		} else if mode == pdf417Lower {
			if (v & pdf417Alpha) != 0 {
				if (enc.getTextTypeAndValue(length, k+1) &
					enc.getTextTypeAndValue(length, k+2) & pdf417Alpha) != 0 {
					dest[ptr] = pdf417ML
					ptr++
					dest[ptr] = pdf417AL
					ptr++
					mode = pdf417Alpha
				} else {
					dest[ptr] = pdf417AS
					ptr++
				}
				dest[ptr] = v & 0xFF
				ptr++
			} else if (v & pdf417Mixed) != 0 {
				dest[ptr] = pdf417ML
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Mixed
			} else if (enc.getTextTypeAndValue(length, k+1) &
				enc.getTextTypeAndValue(length, k+2) & pdf417Punctuation) != 0 {
				dest[ptr] = pdf417ML
				ptr++
				dest[ptr] = pdf417PL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Punctuation
			} else {
				dest[ptr] = pdf417PS
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
			}
		} else if mode == pdf417Mixed {
			if (v & pdf417Lower) != 0 {
				dest[ptr] = pdf417LL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Lower
			} else if (v & pdf417Alpha) != 0 {
				dest[ptr] = pdf417AL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Alpha
			} else if (enc.getTextTypeAndValue(length, k+1) &
				enc.getTextTypeAndValue(length, k+2) & pdf417Punctuation) != 0 {
				dest[ptr] = pdf417PL
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
				mode = pdf417Punctuation
			} else {
				dest[ptr] = pdf417PS
				ptr++
				dest[ptr] = v & 0xFF
				ptr++
			}
		} else if mode == pdf417Punctuation {
			dest[ptr] = pdf417PAL
			ptr++
			mode = pdf417Alpha
			k-- // re-process this character
		}

		k++
	}

	if (ptr & 1) != 0 {
		dest[ptr] = pdf417PS
		ptr++
	}
	size := (ptr + fullBytes) / 2
	if size+enc.cwPtr > pdf417MaxDataCodewords {
		return fmt.Errorf("the text is too big")
	}
	lengthEnd := ptr
	ptr = 0
	for ptr < lengthEnd {
		v := dest[ptr]
		ptr++
		if v >= 30 {
			enc.codewords[enc.cwPtr] = v
			enc.cwPtr++
			enc.codewords[enc.cwPtr] = dest[ptr]
			enc.cwPtr++
			ptr++
		} else {
			enc.codewords[enc.cwPtr] = v*30 + dest[ptr]
			enc.cwPtr++
			ptr++
		}
	}
	return nil
}

// ---------- Number compaction ----------

func (enc *pdf417Encoder) basicNumberCompaction(start, length int) {
	ret := enc.cwPtr
	retLast := length / 3
	enc.cwPtr += retLast + 1
	for k := 0; k <= retLast; k++ {
		enc.codewords[ret+k] = 0
	}
	enc.codewords[ret+retLast] = 1
	end := length + start
	for ni := start; ni < end; ni++ {
		for k := retLast; k >= 0; k-- {
			enc.codewords[ret+k] *= 10
		}
		enc.codewords[ret+retLast] += int(enc.text[ni]) - int('0')
		for k := retLast; k > 0; k-- {
			enc.codewords[ret+k-1] += enc.codewords[ret+k] / 900
			enc.codewords[ret+k] %= 900
		}
	}
}

func (enc *pdf417Encoder) numberCompaction(start, length int) error {
	full := (length / 44) * 15
	sz := length % 44
	var size int
	if sz == 0 {
		size = full
	} else {
		size = full + sz/3 + 1
	}
	if size+enc.cwPtr > pdf417MaxDataCodewords {
		return fmt.Errorf("the text is too big")
	}
	end := length + start
	k := start
	for k < end {
		segSz := end - k
		if segSz > 44 {
			segSz = 44
		}
		enc.basicNumberCompaction(k, segSz)
		k += 44
	}
	return nil
}

// ---------- Byte compaction ----------

func (enc *pdf417Encoder) byteCompaction6(start int) {
	length := 6
	ret := enc.cwPtr
	retLast := 4
	enc.cwPtr += retLast + 1
	for k := 0; k <= retLast; k++ {
		enc.codewords[ret+k] = 0
	}
	end := length + start
	for ni := start; ni < end; ni++ {
		for k := retLast; k >= 0; k-- {
			enc.codewords[ret+k] *= 256
		}
		enc.codewords[ret+retLast] += int(enc.text[ni]) & 0xFF
		for k := retLast; k > 0; k-- {
			enc.codewords[ret+k-1] += enc.codewords[ret+k] / 900
			enc.codewords[ret+k] %= 900
		}
	}
}

func (enc *pdf417Encoder) byteCompaction(start, length int) error {
	size := (length/6)*5 + (length % 6)
	if size+enc.cwPtr > pdf417MaxDataCodewords {
		return fmt.Errorf("the text is too big")
	}
	end := length + start
	k := start
	for k < end {
		sz := end - k
		if sz > 6 {
			sz = 6
		}
		if sz < 6 {
			for j := 0; j < sz; j++ {
				enc.codewords[enc.cwPtr] = int(enc.text[k+j]) & 0xFF
				enc.cwPtr++
			}
		} else {
			enc.byteCompaction6(k)
		}
		k += 6
	}
	return nil
}

// ---------- Break string ----------

func (enc *pdf417Encoder) breakString() {
	textLength := len(enc.text)
	lastP := 0
	startN := 0
	nd := 0

	for k := 0; k < textLength; k++ {
		c := enc.text[k] & 0xFF
		ch := rune(c)
		if ch >= '0' && ch <= '9' {
			if nd == 0 {
				startN = k
			}
			nd++
			continue
		}
		if nd >= 13 {
			if lastP != startN {
				c2 := enc.text[lastP] & 0xFF
				ch2 := rune(c2)
				lastTxt := (ch2 >= ' ' && ch2 < 0x7f) || ch2 == '\r' || ch2 == '\n' || ch2 == '\t'
				for j := lastP; j < startN; j++ {
					c2 = enc.text[j] & 0xFF
					ch2 = rune(c2)
					txt := (ch2 >= ' ' && ch2 < 0x7f) || ch2 == '\r' || ch2 == '\n' || ch2 == '\t'
					if txt != lastTxt {
						segT := byte('B')
						if lastTxt {
							segT = 'T'
						}
						enc.segmentList = append(enc.segmentList, pdf417Segment{segT, lastP, j})
						lastP = j
						lastTxt = txt
					}
				}
				segT := byte('B')
				if lastTxt {
					segT = 'T'
				}
				enc.segmentList = append(enc.segmentList, pdf417Segment{segT, lastP, startN})
			}
			enc.segmentList = append(enc.segmentList, pdf417Segment{'N', startN, k})
			lastP = k
		}
		nd = 0
	}

	if nd < 13 {
		startN = textLength
	}
	if lastP != startN {
		c2 := enc.text[lastP] & 0xFF
		ch2 := rune(c2)
		lastTxt := (ch2 >= ' ' && ch2 < 0x7f) || ch2 == '\r' || ch2 == '\n' || ch2 == '\t'
		for j := lastP; j < startN; j++ {
			c2 = enc.text[j] & 0xFF
			ch2 = rune(c2)
			txt := (ch2 >= ' ' && ch2 < 0x7f) || ch2 == '\r' || ch2 == '\n' || ch2 == '\t'
			if txt != lastTxt {
				segT := byte('B')
				if lastTxt {
					segT = 'T'
				}
				enc.segmentList = append(enc.segmentList, pdf417Segment{segT, lastP, j})
				lastP = j
				lastTxt = txt
			}
		}
		segT := byte('B')
		if lastTxt {
			segT = 'T'
		}
		enc.segmentList = append(enc.segmentList, pdf417Segment{segT, lastP, startN})
	}
	if nd >= 13 {
		enc.segmentList = append(enc.segmentList, pdf417Segment{'N', startN, textLength})
	}

	// Merge pass 1: single-byte segments between text segments
	k := 0
	for k < len(enc.segmentList) {
		v := &enc.segmentList[k]
		var vp *pdf417Segment
		if k > 0 {
			vp = &enc.segmentList[k-1]
		}
		var vn *pdf417Segment
		if k+1 < len(enc.segmentList) {
			vn = &enc.segmentList[k+1]
		}
		if v.segType == 'B' && (v.end-v.start) == 1 {
			if vp != nil && vn != nil &&
				vp.segType == 'T' && vn.segType == 'T' &&
				(vp.end-vp.start)+(vn.end-vn.start) >= 3 {
				vp.end = vn.end
				// remove v and vn
				enc.segmentList = append(enc.segmentList[:k], enc.segmentList[k+2:]...)
				k = 0
				continue
			}
		}
		k++
	}

	// Merge pass 2: absorb short neighbors into long text segments
	k = 0
	for k < len(enc.segmentList) {
		v := &enc.segmentList[k]
		if v.segType == 'T' && (v.end-v.start) >= 5 {
			redo := false
			if k > 0 {
				vp := &enc.segmentList[k-1]
				if (vp.segType == 'B' && (vp.end-vp.start) == 1) || vp.segType == 'T' {
					redo = true
					v.start = vp.start
					enc.segmentList = append(enc.segmentList[:k-1], enc.segmentList[k:]...)
					k--
					v = &enc.segmentList[k]
				}
			}
			if k+1 < len(enc.segmentList) {
				vn := &enc.segmentList[k+1]
				if (vn.segType == 'B' && (vn.end-vn.start) == 1) || vn.segType == 'T' {
					redo = true
					v.end = vn.end
					enc.segmentList = append(enc.segmentList[:k+1], enc.segmentList[k+2:]...)
				}
			}
			if redo {
				k = 0
				continue
			}
		}
		k++
	}

	// Merge pass 3: absorb short text segments into byte segments
	k = 0
	for k < len(enc.segmentList) {
		v := &enc.segmentList[k]
		if v.segType == 'B' {
			redo := false
			if k > 0 {
				vp := &enc.segmentList[k-1]
				if (vp.segType == 'T' && (vp.end-vp.start) < 5) || vp.segType == 'B' {
					redo = true
					v.start = vp.start
					enc.segmentList = append(enc.segmentList[:k-1], enc.segmentList[k:]...)
					k--
					v = &enc.segmentList[k]
				}
			}
			if k+1 < len(enc.segmentList) {
				vn := &enc.segmentList[k+1]
				if (vn.segType == 'T' && (vn.end-vn.start) < 5) || vn.segType == 'B' {
					redo = true
					v.end = vn.end
					enc.segmentList = append(enc.segmentList[:k+1], enc.segmentList[k+2:]...)
				}
			}
			if redo {
				k = 0
				continue
			}
		}
		k++
	}

	// Special case: single text segment of all digits
	if len(enc.segmentList) == 1 {
		v := &enc.segmentList[0]
		if v.segType == 'T' && (v.end-v.start) >= 8 {
			allDigits := true
			for kk := v.start; kk < v.end; kk++ {
				ch := rune(enc.text[kk] & 0xFF)
				if ch < '0' || ch > '9' {
					allDigits = false
					break
				}
			}
			if allDigits {
				v.segType = 'N'
			}
		}
	}
}

// ---------- Assemble ----------

func (enc *pdf417Encoder) assemble() error {
	if len(enc.segmentList) == 0 {
		return nil
	}
	enc.cwPtr = 1
	for k, v := range enc.segmentList {
		segLen := v.end - v.start
		switch v.segType {
		case 'T':
			if k != 0 {
				enc.codewords[enc.cwPtr] = pdf417TextMode
				enc.cwPtr++
			}
			if err := enc.textCompaction(v.start, segLen); err != nil {
				return err
			}
		case 'N':
			enc.codewords[enc.cwPtr] = pdf417NumericMode
			enc.cwPtr++
			if err := enc.numberCompaction(v.start, segLen); err != nil {
				return err
			}
		case 'B':
			if segLen%6 != 0 {
				enc.codewords[enc.cwPtr] = pdf417ByteMode
			} else {
				enc.codewords[enc.cwPtr] = pdf417ByteMode6
			}
			enc.cwPtr++
			if err := enc.byteCompaction(v.start, segLen); err != nil {
				return err
			}
		}
	}
	return nil
}

// ---------- Max possible error level ----------

func (enc *pdf417Encoder) maxPossibleErrorLevel(remain int) int {
	level := 8
	size := 512
	for level > 0 {
		if remain >= size {
			return level
		}
		level--
		size >>= 1
	}
	return 0
}

// ---------- Max square ----------

func (enc *pdf417Encoder) maxSquare() int {
	if enc.codeColumns > 21 {
		enc.codeColumns = 29
		enc.codeRows = 32
	} else {
		enc.codeColumns = 16
		enc.codeRows = 58
	}
	return pdf417MaxDataCodewords + 2
}

// ---------- Paint code (main entry) ----------

func (enc *pdf417Encoder) paintCode() error {
	enc.initBlock()

	if len(enc.text) == 0 {
		return fmt.Errorf("text cannot be empty")
	}
	if len(enc.text) > pdf417AbsoluteMaxTextSize {
		return fmt.Errorf("the text is too big")
	}

	enc.segmentList = nil
	enc.breakString()
	if err := enc.assemble(); err != nil {
		return err
	}
	enc.segmentList = nil
	enc.codewords[0] = enc.cwPtr
	enc.lenCodewords = enc.cwPtr

	maxErr := enc.maxPossibleErrorLevel(pdf417MaxDataCodewords + 2 - enc.lenCodewords)

	if enc.useAutoErrorLevel {
		if enc.lenCodewords < 41 {
			enc.errorLevel = 2
		} else if enc.lenCodewords < 161 {
			enc.errorLevel = 3
		} else if enc.lenCodewords < 321 {
			enc.errorLevel = 4
		} else {
			enc.errorLevel = 5
		}
	}

	if enc.errorLevel < 0 {
		enc.errorLevel = 0
	} else if enc.errorLevel > maxErr {
		enc.errorLevel = maxErr
	}

	if enc.codeColumns < 1 {
		enc.codeColumns = 1
	} else if enc.codeColumns > 30 {
		enc.codeColumns = 30
	}

	if enc.codeRows < 3 {
		enc.codeRows = 3
	} else if enc.codeRows > 90 {
		enc.codeRows = 90
	}

	lenErr := 2 << enc.errorLevel
	fixedColumn := enc.sizeKind != pdf417Rows
	skipRowColAdjust := false
	tot := enc.lenCodewords + lenErr

	if enc.sizeKind == pdf417ColumnsAndRows {
		tot = enc.codeColumns * enc.codeRows
		if tot > pdf417MaxDataCodewords+2 {
			tot = enc.maxSquare()
		}
		if tot < enc.lenCodewords+lenErr {
			tot = enc.lenCodewords + lenErr
		} else {
			skipRowColAdjust = true
		}
	} else if enc.sizeKind == pdf417Auto {
		fixedColumn = true
		if enc.aspectRatio < 0.001 {
			enc.aspectRatio = 0.001
		} else if enc.aspectRatio > 1000 {
			enc.aspectRatio = 1000
		}
		b := 73*enc.aspectRatio - 4
		c := (-b + math.Sqrt(b*b+4*17*enc.aspectRatio*
			float64(enc.lenCodewords+lenErr)*float64(enc.yHeight))) /
			(2 * 17 * enc.aspectRatio)
		enc.codeColumns = int(c + 0.5)
		if enc.codeColumns < 1 {
			enc.codeColumns = 1
		} else if enc.codeColumns > 30 {
			enc.codeColumns = 30
		}
	}

	if !skipRowColAdjust {
		if fixedColumn {
			enc.codeRows = (tot - 1) / enc.codeColumns + 1
			if enc.codeRows < 3 {
				enc.codeRows = 3
			} else if enc.codeRows > 90 {
				enc.codeRows = 90
				enc.codeColumns = (tot - 1) / 90 + 1
			}
		} else {
			enc.codeColumns = (tot - 1) / enc.codeRows + 1
			if enc.codeColumns > 30 {
				enc.codeColumns = 30
				enc.codeRows = (tot - 1) / 30 + 1
			}
		}
		tot = enc.codeRows * enc.codeColumns
	}

	if tot > pdf417MaxDataCodewords+2 {
		tot = enc.maxSquare()
	}

	enc.errorLevel = enc.maxPossibleErrorLevel(tot - enc.lenCodewords)
	lenErr = 2 << enc.errorLevel
	pad := tot - lenErr - enc.lenCodewords
	enc.cwPtr = enc.lenCodewords
	for pad > 0 {
		enc.codewords[enc.cwPtr] = pdf417TextMode
		enc.cwPtr++
		pad--
	}
	enc.codewords[0] = enc.cwPtr
	enc.lenCodewords = enc.cwPtr
	enc.calculateErrorCorrection(enc.lenCodewords)
	enc.lenCodewords = tot

	enc.outPaintCode()
	return nil
}

// convertToBoolMatrix converts outBits to a [][]bool matrix.
func (enc *pdf417Encoder) convertToBoolMatrix() [][]bool {
	if len(enc.outBits) == 0 || enc.codeRows == 0 || enc.bitColumns == 0 {
		return nil
	}
	rows := enc.codeRows
	bitCols := enc.bitColumns
	bytesPerRow := (bitCols + 7) / 8
	result := make([][]bool, rows)
	for r := 0; r < rows; r++ {
		rowData := make([]bool, bitCols)
		for c := 0; c < bitCols; c++ {
			byteIndex := r*bytesPerRow + c/8
			bitIndex := 7 - (c % 8)
			if byteIndex < len(enc.outBits) {
				rowData[c] = (enc.outBits[byteIndex] & (1 << bitIndex)) != 0
			}
		}
		result[r] = rowData
	}
	return result
}

// ========== Public PDF417 struct ==========

// PDF417 encodes PDF417 2D barcodes.
type PDF417 struct {
	BarcodeBase2D
	errorLevel        int
	columns           int
	rows              int
	sizeMode          int
	aspectRatio       float64
	yHeight           int
	quietZoneWidth    int
	useAutoErrorLevel bool
}

// NewPDF417 creates a new PDF417 encoder.
func NewPDF417(outputFormat string) *PDF417 {
	p := &PDF417{
		errorLevel:        PDF417ErrorLevel2,
		columns:           0,
		rows:              0,
		sizeMode:          PDF417SizeAuto,
		aspectRatio:       0.5,
		yHeight:           3,
		quietZoneWidth:    2,
		useAutoErrorLevel: true,
	}
	p.InitBase2D(outputFormat)
	return p
}

// --- Settings ---

// SetErrorLevel sets the error correction level (-1=Auto, 0..8).
func (p *PDF417) SetErrorLevel(level int) {
	if level >= PDF417ErrorAuto && level <= PDF417ErrorLevel8 {
		p.errorLevel = level
	}
}

// ErrorLevel returns the current error correction level.
func (p *PDF417) ErrorLevel() int { return p.errorLevel }

// SetColumns sets the number of data columns (0=auto, 1..30).
func (p *PDF417) SetColumns(cols int) {
	if cols >= 0 && cols <= 30 {
		p.columns = cols
	}
}

// Columns returns the current column count.
func (p *PDF417) Columns() int { return p.columns }

// SetRows sets the number of rows (0=auto, 3..90).
func (p *PDF417) SetRows(rows int) {
	if rows >= 0 && rows <= 90 {
		p.rows = rows
	}
}

// Rows returns the current row count.
func (p *PDF417) Rows() int { return p.rows }

// SetSizeMode sets the size mode (PDF417SizeAuto, PDF417SizeRows, PDF417SizeColumns, PDF417SizeColumnsAndRows).
func (p *PDF417) SetSizeMode(mode int) {
	if mode >= PDF417SizeAuto && mode <= PDF417SizeColumnsAndRows {
		p.sizeMode = mode
	}
}

// SizeMode returns the current size mode.
func (p *PDF417) SizeMode() int { return p.sizeMode }

// SetAspectRatio sets the aspect ratio (0.001..1000).
func (p *PDF417) SetAspectRatio(ratio float64) {
	if ratio >= 0.001 && ratio <= 1000.0 {
		p.aspectRatio = ratio
	}
}

// AspectRatio returns the current aspect ratio.
func (p *PDF417) AspectRatio() float64 { return p.aspectRatio }

// SetYHeight sets the Y-dimension height multiplier.
func (p *PDF417) SetYHeight(h int) {
	if h > 0 {
		p.yHeight = h
	}
}

// YHeight returns the current Y-height.
func (p *PDF417) YHeight() int { return p.yHeight }

// SetQuietZoneWidth sets the quiet zone width in modules.
func (p *PDF417) SetQuietZoneWidth(w int) {
	if w >= 0 {
		p.quietZoneWidth = w
	}
}

// QuietZoneWidth returns the current quiet zone width.
func (p *PDF417) QuietZoneWidth() int { return p.quietZoneWidth }

// SetUseAutoErrorLevel sets whether to automatically determine error level.
func (p *PDF417) SetUseAutoErrorLevel(on bool) {
	p.useAutoErrorLevel = on
}

// UseAutoErrorLevel returns the current auto error level setting.
func (p *PDF417) UseAutoErrorLevel() bool { return p.useAutoErrorLevel }

// --- Internal: apply settings to encoder ---

func (p *PDF417) applySettingsCal(enc *pdf417Encoder) {
	enc.errorLevel = p.errorLevel
	enc.codeColumns = p.columns
	enc.codeRows = p.rows
	enc.aspectRatio = p.aspectRatio
	enc.yHeight = p.yHeight
	enc.useAutoErrorLevel = p.useAutoErrorLevel
}

func (p *PDF417) applySettingsFull(enc *pdf417Encoder) {
	enc.errorLevel = p.errorLevel
	enc.codeColumns = p.columns
	enc.codeRows = p.rows
	enc.aspectRatio = p.aspectRatio
	enc.yHeight = p.yHeight
	enc.useAutoErrorLevel = p.useAutoErrorLevel
	switch p.sizeMode {
	case PDF417SizeAuto:
		enc.sizeKind = pdf417Auto
	case PDF417SizeColumns:
		enc.sizeKind = pdf417Columns
	case PDF417SizeRows:
		enc.sizeKind = pdf417Rows
	case PDF417SizeColumnsAndRows:
		enc.sizeKind = pdf417ColumnsAndRows
	}
}

// --- GetPattern ---

// GetPattern generates the PDF417 pattern as a [][]bool matrix (true=black).
func (p *PDF417) GetPattern(code string) ([][]bool, error) {
	data := []byte(code)
	if len(data) == 0 {
		return nil, fmt.Errorf("empty input")
	}

	enc := newPDF417Encoder()
	p.applySettingsCal(enc)
	enc.text = data
	if err := enc.paintCode(); err != nil {
		return nil, err
	}
	result := enc.convertToBoolMatrix()
	if result == nil {
		return nil, fmt.Errorf("failed to generate pattern")
	}
	return result, nil
}

// --- Draw ---

// Draw renders the PDF417 barcode to the internal buffer (SVG or PNG/JPEG).
func (p *PDF417) Draw(code string, width, height int) error {
	if width <= 0 || height <= 0 {
		return fmt.Errorf("width and height must be positive")
	}

	patt, err := p.GetPattern(code)
	if err != nil {
		return err
	}

	numRows := len(patt)
	if numRows == 0 {
		return fmt.Errorf("empty pattern")
	}
	numCols := len(patt[0])
	if numCols == 0 {
		return fmt.Errorf("empty pattern")
	}

	if p.IsSVGOutput() {
		return p.drawSVGPDF417(patt, numRows, numCols, width, height)
	}
	return p.drawPNGPDF417(patt, numRows, numCols, width, height)
}

// drawSVGPDF417 renders the PDF417 pattern as SVG.
func (p *PDF417) drawSVGPDF417(patt [][]bool, numRows, numCols, width, height int) error {
	var moduleW, moduleH float64
	if p.fitWidth {
		moduleW = float64(width) / float64(numCols)
		moduleH = float64(height) / float64(numRows)
	} else {
		moduleW = float64(width / numCols)
		moduleH = float64(height / numRows)
		if moduleW < 1.0 {
			moduleW = 1.0
		}
		if moduleH < 1.0 {
			moduleH = 1.0
		}
	}

	aw := int(math.Ceil(moduleW * float64(numCols)))
	ah := int(math.Ceil(moduleH * float64(numRows)))

	p.svgBegin(aw, ah)
	p.svgRect(0, 0, float64(aw), float64(ah), p.backColor)

	adj := float64(p.pxAdjBlack)
	for r := 0; r < numRows; r++ {
		for c := 0; c < numCols; c++ {
			if patt[r][c] {
				dw := moduleW + adj + 0.5
				dh := moduleH + adj + 0.5
				if dw < 0 {
					dw = 0
				}
				if dh < 0 {
					dh = 0
				}
				p.svgRect(float64(c)*moduleW, float64(r)*moduleH, dw, dh, p.foreColor)
			}
		}
	}

	p.svgEnd()
	return nil
}

// drawPNGPDF417 renders the PDF417 pattern as PNG/JPEG.
func (p *PDF417) drawPNGPDF417(patt [][]bool, numRows, numCols, width, height int) error {
	var moduleW, moduleH float64
	if p.fitWidth {
		moduleW = float64(width) / float64(numCols)
		moduleH = float64(height) / float64(numRows)
	} else {
		moduleW = float64(width / numCols)
		moduleH = float64(height / numRows)
		if moduleW < 1.0 {
			moduleW = 1.0
		}
		if moduleH < 1.0 {
			moduleH = 1.0
		}
	}

	aw := int(math.Ceil(moduleW * float64(numCols)))
	ah := int(math.Ceil(moduleH * float64(numRows)))

	img := image.NewNRGBA(image.Rect(0, 0, aw, ah))
	draw.Draw(img, img.Bounds(), &image.Uniform{p.backColor}, image.Point{}, draw.Src)

	adj := p.pxAdjBlack
	for r := 0; r < numRows; r++ {
		for c := 0; c < numCols; c++ {
			if patt[r][c] {
				drawX := int(float64(c) * moduleW)
				drawY := int(float64(r) * moduleH)
				drawW := int(moduleW + float64(adj))
				drawH := int(moduleH + float64(adj))
				if drawW < 1 {
					drawW = 1
				}
				if drawH < 1 {
					drawH = 1
				}
				endX := drawX + drawW
				endY := drawY + drawH
				if endX > aw {
					endX = aw
				}
				if endY > ah {
					endY = ah
				}
				fillRect(img, drawX, drawY, endX, endY, p.foreColor)
			}
		}
	}

	// Trial mode watermark
	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, aw, ah)
	}

	return p.encodeImage(img)
}
