package barcode

import (
	"fmt"
	"strings"
)

var (
	nw7Chars    = "ABCD.+:/$-0123456789"
	nw7Patterns = []int{
		0x1a, 0x29, 0x0b, 0x0e, 0x54, 0x15, 0x45, 0x51, 0x18, 0x0c,
		0x03, 0x06, 0x09, 0x60, 0x12, 0x42, 0x21, 0x24, 0x30, 0x48,
	}
)

// NW7 (Codabar) encoder.
type NW7 struct {
	BarcodeBase1D
	ShowStartStop bool
}

// NewNW7 creates a new NW7 (Codabar) encoder.
func NewNW7(outputFormat string) *NW7 {
	n := &NW7{ShowStartStop: true}
	n.InitBase1D(outputFormat)
	return n
}

// Encode returns the bar/space width pattern for NW7.
func (n *NW7) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	s := strings.ToUpper(code)

	startChar := s[0]
	stopChar := s[len(s)-1]
	isStartValid := startChar >= 'A' && startChar <= 'D'
	isStopValid := stopChar >= 'A' && stopChar <= 'D'

	var encoded string
	if isStartValid && isStopValid {
		encoded = s
	} else if isStartValid || isStopValid {
		return nil, fmt.Errorf("invalid start/stop character combination in NW7: %s", s)
	} else {
		encoded = "C" + s + "C"
	}

	var result []int
	for i := 0; i < len(encoded); i++ {
		if i > 0 {
			result = append(result, 2) // inter-character gap
		}

		charIndex := strings.IndexByte(nw7Chars, encoded[i])
		if charIndex == -1 {
			return nil, fmt.Errorf("invalid character in NW7: %c", encoded[i])
		}

		pattern := nw7Patterns[charIndex]

		// 7-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
		mask := 0x40
		for j := 0; j < 7; j++ {
			if (pattern & (mask >> j)) != 0 {
				result = append(result, 3)
			} else {
				result = append(result, 1)
			}
		}
	}

	return result, nil
}

// Draw renders the NW7 barcode.
func (n *NW7) Draw(code string, width, height int) error {
	return n.BarcodeBase1D.Draw(code, width, height, n)
}
