package barcode

import "fmt"

var (
	nec2of5DigitPatterns = []int{
		0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
	}
	nec2of5Start = 0x5 // 101
	nec2of5Stop  = 0x3 // 011
)

// NEC2of5 encoder.
type NEC2of5 struct {
	BarcodeBase1D
}

// NewNEC2of5 creates a new NEC 2 of 5 encoder.
func NewNEC2of5(outputFormat string) *NEC2of5 {
	n := &NEC2of5{}
	n.InitBase1D(outputFormat)
	return n
}

// Encode returns the bar/space width pattern for NEC 2 of 5.
func (n *NEC2of5) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("NEC 2 of 5 barcode requires numeric digits only")
		}
	}

	wide := 2
	if n.minLineWidth%2 != 0 {
		wide = 3
	}
	narrow := 1

	var result []int

	// Start code (3 bits: 101)
	chk := 0x4
	for j := 0; j < 3; j++ {
		if j%2 == 0 { // bar position
			if (nec2of5Start & (chk >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		} else { // space position - always narrow for NEC
			result = append(result, narrow)
		}
	}

	// Inter-character gap
	result = append(result, narrow)

	// Data encoding
	for i := 0; i < len(code); i++ {
		digit := int(code[i] - '0')
		pattern := nec2of5DigitPatterns[digit]

		chk := 0x10
		for j := 0; j < 5; j++ {
			if j%2 == 0 { // bar position
				if (pattern & (chk >> j)) != 0 {
					result = append(result, wide)
				} else {
					result = append(result, narrow)
				}
			} else { // space position - always narrow for NEC data
				result = append(result, narrow)
			}
		}

		// Inter-character gap (except last)
		if i < len(code)-1 {
			result = append(result, narrow)
		}
	}

	// Stop code (3 bits: 011) - both bar and space respond to bit pattern
	chk = 0x4
	for j := 0; j < 3; j++ {
		if j%2 == 0 { // bar position
			if (nec2of5Stop & (chk >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		} else { // space position - responds to bit pattern for stop
			if (nec2of5Stop & (chk >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		}
	}

	return result, nil
}

// Draw renders the NEC 2 of 5 barcode.
func (n *NEC2of5) Draw(code string, width, height int) error {
	return n.BarcodeBase1D.Draw(code, width, height, n)
}
