package barcode

import "fmt"

var (
	matrix2of5DigitPatterns = []int{
		0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
	}
	matrix2of5Start = 0x5 // 101
	matrix2of5Stop  = 0x3 // 011
)

// Matrix2of5 encoder.
type Matrix2of5 struct {
	BarcodeBase1D
}

// NewMatrix2of5 creates a new Matrix 2 of 5 encoder.
func NewMatrix2of5(outputFormat string) *Matrix2of5 {
	m := &Matrix2of5{}
	m.InitBase1D(outputFormat)
	return m
}

// Encode returns the bar/space width pattern for Matrix 2 of 5.
func (m *Matrix2of5) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("Matrix 2 of 5 barcode requires numeric digits only")
		}
	}

	wide := 2
	if m.minLineWidth%2 != 0 {
		wide = 3
	}
	narrow := 1

	var result []int

	// Start code (3 bits: 101)
	mask := 0x4
	for j := 0; j < 3; j++ {
		if j%2 == 0 { // bar position
			if (matrix2of5Start & (mask >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		} else { // space position
			result = append(result, narrow)
		}
	}

	// Inter-character gap
	result = append(result, narrow)

	// Data encoding
	for i := 0; i < len(code); i++ {
		digit := int(code[i] - '0')
		pattern := matrix2of5DigitPatterns[digit]

		bitMask := 0x10
		for j := 0; j < 5; j++ {
			if j%2 == 0 { // bar position
				if (pattern & (bitMask >> j)) != 0 {
					result = append(result, wide)
				} else {
					result = append(result, narrow)
				}
			} else { // space position - always narrow
				result = append(result, narrow)
			}
		}

		// Inter-character gap (except last)
		if i < len(code)-1 {
			result = append(result, narrow)
		}
	}

	// Stop code (3 bits: 011)
	mask = 0x4
	for j := 0; j < 3; j++ {
		if j%2 == 0 { // bar position
			if (matrix2of5Stop & (mask >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		} else { // space position - responds to bit pattern for stop
			if (matrix2of5Stop & (mask >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, narrow)
			}
		}
	}

	return result, nil
}

// Draw renders the Matrix 2 of 5 barcode.
func (m *Matrix2of5) Draw(code string, width, height int) error {
	return m.BarcodeBase1D.Draw(code, width, height, m)
}
