package barcode

// JAN/UPC extended guard bar SVG rendering.
// Ported from Python barcode_base_1d.py _draw_svg_jan_upc.

import (
	"fmt"
	"image"
	"image/draw"
	"math"
)

// JANUPCParams holds shared text/bar parameters for JAN/UPC rendering.
type JANUPCParams struct {
	TextHeight      float64
	NormalBarHeight float64
	TallBarHeight   float64
	SVGFontSize     int
	TextY           float64
	DigitWidth      float64
	LeftMargin      float64
	CharWidth       float64
	BarcodeWidth    float64
	BarcodeStartX   float64
	ModuleWidth     float64
	PosA            float64
}

// ComputeJANUPCParams computes shared text/bar parameters for JAN/UPC rendering.
func ComputeJANUPCParams(height int, tfScale, tvoScale float64, pattern []int, width int, clsName string) JANUPCParams {
	var p JANUPCParams

	baseRatio := 0.2
	if height < 50 {
		nr := baseRatio * 1.25
		if nr < 0.25 {
			baseRatio = nr
		} else {
			baseRatio = 0.25
		}
	} else if height > 200 {
		nr := baseRatio * 0.75
		if nr > 0.15 {
			baseRatio = nr
		} else {
			baseRatio = 0.15
		}
	}

	p.TextHeight = float64(int(float64(height) * baseRatio * tfScale))
	if p.TextHeight < 8 {
		p.TextHeight = 8
	}

	voRatio := 0.05
	if height < 50 {
		nr := voRatio * 0.6
		if nr > 0.03 {
			voRatio = nr
		} else {
			voRatio = 0.03
		}
	} else if height > 200 {
		nr := voRatio * 1.4
		if nr < 0.08 {
			voRatio = nr
		} else {
			voRatio = 0.08
		}
	}

	verticalOffsetBar := float64(int(float64(height) * voRatio * tvoScale))
	if verticalOffsetBar < 2 {
		verticalOffsetBar = 2
	}

	p.NormalBarHeight = float64(height) - p.TextHeight - verticalOffsetBar
	if p.NormalBarHeight < 1 {
		p.NormalBarHeight = 1
	}

	p.SVGFontSize = int(p.TextHeight * 0.8)
	if p.SVGFontSize < 8 {
		p.SVGFontSize = 8
	}

	verticalOffsetText := float64(int(float64(height) * 0.03 * tvoScale))
	if verticalOffsetText < 1 {
		verticalOffsetText = 1
	}
	p.TextY = p.NormalBarHeight + verticalOffsetText

	p.DigitWidth = float64(p.SVGFontSize) * 0.6
	p.LeftMargin = p.DigitWidth * 1.0

	fontHeight := float64(height) * 0.2 * tfScale
	p.CharWidth = fontHeight * 0.8

	var longBarExtra float64
	if clsName == "JAN8" || clsName == "JAN13" {
		longBarExtra = fontHeight * 0.275
	} else {
		longBarExtra = fontHeight * 0.495
	}
	p.TallBarHeight = p.NormalBarHeight + longBarExtra

	p.PosA = 0

	totalUnits := 0
	for _, v := range pattern {
		totalUnits += v
	}

	switch clsName {
	case "JAN8":
		p.BarcodeWidth = float64(width)
		p.BarcodeStartX = 0
	case "JAN13":
		leftMarginWidth := p.CharWidth * 1.2
		p.BarcodeWidth = float64(width) - leftMarginWidth
		p.BarcodeStartX = leftMarginWidth
	case "UPCA", "UPCE":
		outMargin := p.CharWidth * 1.0
		p.BarcodeWidth = float64(width) - outMargin*2
		p.BarcodeStartX = outMargin
	default:
		p.BarcodeWidth = float64(width)
		p.BarcodeStartX = 0
	}

	if totalUnits > 0 {
		p.ModuleWidth = p.BarcodeWidth / float64(totalUnits)
	}

	return p
}

// drawSVGJAN8 renders JAN8 with extended guard bars as SVG.
func (b *BarcodeBase1D) drawSVGJAN8(pattern []int, displayCode string, width, height int) error {
	p := ComputeJANUPCParams(height, b.textFontScale, b.textVerticalOffsetScale, pattern, width, "JAN8")

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	currentX := p.BarcodeStartX
	isBar := true
	segIdx, segCnt := 0, 0
	var posB, posC, posD, posE float64

	for _, v := range pattern {
		barW := float64(v) * p.ModuleWidth
		if barW < 1 {
			barW = 1
		}

		if segIdx == 0 && segCnt >= 3 {
			posB = currentX; segIdx++; segCnt = 0
		} else if segIdx == 1 && segCnt >= 16 {
			posC = currentX; segIdx++; segCnt = 0
		} else if segIdx == 2 && segCnt >= 5 {
			posD = currentX; segIdx++; segCnt = 0
		} else if segIdx == 3 && segCnt >= 16 {
			posE = currentX; segIdx++; segCnt = 0
		}

		isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
		if isBar {
			bh := p.NormalBarHeight
			if isTall {
				bh = p.TallBarHeight
			}
			b.svgRect(currentX, 0, barW, bh, b.foreColor)
		}
		segCnt++
		currentX += barW
		isBar = !isBar
	}

	// Fallback
	mw := p.ModuleWidth
	if posB == 0 { posB = p.BarcodeStartX + 3*mw }
	if posC == 0 { posC = posB + 28*mw }
	if posD == 0 { posD = posC + 5*mw }
	if posE == 0 { posE = posD + 28*mw }

	if len(displayCode) >= 8 {
		leftSection := posC - posB
		effLeft := leftSection - p.LeftMargin*2
		if effLeft > 0 {
			if b.TextEvenSpacing {
				spacing := effLeft / 3.0
				for i := 0; i < 4; i++ {
					digitX := posB + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posB + leftSection/2.0
				b.svgText(centerX, p.TextY, displayCode[:4], p.SVGFontSize, b.foreColor, "middle")
			}
		}

		rightSection := posE - posD
		effRight := rightSection - p.LeftMargin*2
		if effRight > 0 {
			if b.TextEvenSpacing {
				spacing := effRight / 3.0
				for i := 0; i < 4; i++ {
					digitX := posD + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[4+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posD + rightSection/2.0
				b.svgText(centerX, p.TextY, displayCode[4:8], p.SVGFontSize, b.foreColor, "middle")
			}
		}
	}

	b.svgEnd()
	return nil
}

// drawSVGJAN13 renders JAN13 with extended guard bars as SVG.
func (b *BarcodeBase1D) drawSVGJAN13(pattern []int, displayCode string, width, height int) error {
	p := ComputeJANUPCParams(height, b.textFontScale, b.textVerticalOffsetScale, pattern, width, "JAN13")

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	currentX := p.BarcodeStartX
	isBar := true
	segIdx, segCnt := 0, 0
	var posB, posC, posD, posE float64

	for _, v := range pattern {
		barW := float64(v) * p.ModuleWidth
		if barW < 1 {
			barW = 1
		}

		if segIdx == 0 && segCnt >= 3 {
			posB = currentX; segIdx++; segCnt = 0
		} else if segIdx == 1 && segCnt >= 24 {
			posC = currentX; segIdx++; segCnt = 0
		} else if segIdx == 2 && segCnt >= 5 {
			posD = currentX; segIdx++; segCnt = 0
		} else if segIdx == 3 && segCnt >= 24 {
			posE = currentX; segIdx++; segCnt = 0
		}

		isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
		if isBar {
			bh := p.NormalBarHeight
			if isTall {
				bh = p.TallBarHeight
			}
			b.svgRect(currentX, 0, barW, bh, b.foreColor)
		}
		segCnt++
		currentX += barW
		isBar = !isBar
	}

	if len(displayCode) >= 13 {
		// Prefix
		prefixX := p.PosA + p.DigitWidth*0.3
		b.svgText(prefixX, p.TextY, string(displayCode[0]), p.SVGFontSize, b.foreColor, "start")

		leftSection := posC - posB
		effLeft := leftSection - p.LeftMargin*2
		if effLeft > 0 {
			if b.TextEvenSpacing {
				spacing := effLeft / 5.0
				for i := 0; i < 6; i++ {
					digitX := posB + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[1+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posB + leftSection/2.0
				b.svgText(centerX, p.TextY, displayCode[1:7], p.SVGFontSize, b.foreColor, "middle")
			}
		}

		rightSection := posE - posD
		effRight := rightSection - p.LeftMargin*2
		if effRight > 0 {
			if b.TextEvenSpacing {
				spacing := effRight / 5.0
				for i := 0; i < 6; i++ {
					digitX := posD + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[7+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posD + rightSection/2.0
				b.svgText(centerX, p.TextY, displayCode[7:13], p.SVGFontSize, b.foreColor, "middle")
			}
		}
	}

	b.svgEnd()
	return nil
}

// drawSVGUPCA renders UPC-A with extended guard bars as SVG.
func (b *BarcodeBase1D) drawSVGUPCA(pattern []int, displayCode string, width, height int) error {
	p := ComputeJANUPCParams(height, b.textFontScale, b.textVerticalOffsetScale, pattern, width, "UPCA")

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	currentX := p.BarcodeStartX
	isBar := true
	segIdx, segCnt := 0, 0
	var posB, posC, posD, posE float64

	for _, v := range pattern {
		barW := float64(v) * p.ModuleWidth
		if barW < 1 {
			barW = 1
		}

		if segIdx == 0 && segCnt >= 7 {
			posB = currentX; segIdx++; segCnt = 0
		} else if segIdx == 1 && segCnt >= 21 {
			posC = currentX; segIdx++; segCnt = 0
		} else if segIdx == 2 && segCnt >= 3 {
			posD = currentX; segIdx++; segCnt = 0
		} else if segIdx == 3 && segCnt >= 21 {
			posE = currentX; segIdx++; segCnt = 0
		}

		isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
		if isBar {
			bh := p.NormalBarHeight
			if isTall {
				bh = p.TallBarHeight
			}
			b.svgRect(currentX, 0, barW, bh, b.foreColor)
		}
		segCnt++
		currentX += barW
		isBar = !isBar
	}

	posF := currentX

	if len(displayCode) >= 12 {
		// Prefix
		prefixX := p.PosA + p.DigitWidth*0.3
		b.svgText(prefixX, p.TextY, string(displayCode[0]), p.SVGFontSize, b.foreColor, "start")

		// Left 5
		leftSection := posC - posB
		effLeft := leftSection - p.LeftMargin*2
		if effLeft > 0 {
			if b.TextEvenSpacing {
				spacing := effLeft / 4.0
				for i := 0; i < 5; i++ {
					digitX := posB + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[1+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posB + leftSection/2.0
				b.svgText(centerX, p.TextY, displayCode[1:6], p.SVGFontSize, b.foreColor, "middle")
			}
		}

		// Right 5
		rightSection := posE - posD
		effRight := rightSection - p.LeftMargin*2
		if effRight > 0 {
			if b.TextEvenSpacing {
				spacing := effRight / 4.0
				for i := 0; i < 5; i++ {
					digitX := posD + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[6+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posD + rightSection/2.0
				b.svgText(centerX, p.TextY, displayCode[6:11], p.SVGFontSize, b.foreColor, "middle")
			}
		}

		// Suffix
		suffixX := posF + p.DigitWidth*0.7
		b.svgText(suffixX, p.TextY, string(displayCode[11]), p.SVGFontSize, b.foreColor, "start")
	}

	b.svgEnd()
	return nil
}

// drawSVGUPCE renders UPC-E with extended guard bars as SVG.
func (b *BarcodeBase1D) drawSVGUPCE(pattern []int, displayCode string, width, height int) error {
	p := ComputeJANUPCParams(height, b.textFontScale, b.textVerticalOffsetScale, pattern, width, "UPCE")

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	currentX := p.BarcodeStartX
	isBar := true
	var posB, posC float64
	accumulatedUnits := 0

	for _, v := range pattern {
		barW := float64(v) * p.ModuleWidth
		if barW < 1 {
			barW = 1
		}

		if posB == 0 && accumulatedUnits >= 3 {
			posB = currentX
		}
		if posC == 0 && accumulatedUnits >= 45 {
			posC = currentX
		}

		isStartGuard := accumulatedUnits < 3
		isEndGuard := accumulatedUnits >= 45
		isTall := (isStartGuard || isEndGuard) && isBar

		if isBar {
			bh := p.NormalBarHeight
			if isTall {
				bh = p.TallBarHeight
			}
			b.svgRect(currentX, 0, barW, bh, b.foreColor)
		}

		accumulatedUnits += v
		currentX += barW
		isBar = !isBar
	}

	if posB == 0 {
		posB = p.BarcodeStartX + 3*p.ModuleWidth
	}
	if posC == 0 {
		posC = p.BarcodeStartX + 45*p.ModuleWidth
	}

	posB += p.CharWidth * 0.5
	posC -= p.CharWidth * 0.5
	posF := currentX

	if len(displayCode) >= 8 {
		// Prefix
		prefixX := p.PosA + p.DigitWidth*0.3
		b.svgText(prefixX, p.TextY, string(displayCode[0]), p.SVGFontSize, b.foreColor, "start")

		midSection := posC - posB
		effMid := midSection - p.LeftMargin*2
		if effMid > 0 {
			if b.TextEvenSpacing {
				spacing := effMid / 5.0
				for i := 0; i < 6; i++ {
					digitX := posB + p.LeftMargin + float64(i)*spacing
					b.svgText(digitX, p.TextY, string(displayCode[1+i]), p.SVGFontSize, b.foreColor, "middle")
				}
			} else {
				centerX := posB + midSection/2.0
				b.svgText(centerX, p.TextY, displayCode[1:7], p.SVGFontSize, b.foreColor, "middle")
			}
		}

		// Suffix
		suffixX := posF + p.DigitWidth*0.7
		b.svgText(suffixX, p.TextY, string(displayCode[7]), p.SVGFontSize, b.foreColor, "start")
	}

	b.svgEnd()
	return nil
}

// drawPNGJANUPC renders JAN/UPC barcodes as PNG with extended guard bars.
func (b *BarcodeBase1D) drawPNGJANUPC(pattern []int, displayCode string, width, height int, clsName string) error {
	p := ComputeJANUPCParams(height, b.textFontScale, b.textVerticalOffsetScale, pattern, width, clsName)

	img := image.NewNRGBA(image.Rect(0, 0, width, height))
	draw.Draw(img, img.Bounds(), &image.Uniform{b.backColor}, image.Point{}, draw.Src)

	currentX := p.BarcodeStartX
	isBar := true

	// Draw bars based on barcode type
	switch clsName {
	case "JAN8":
		segIdx, segCnt := 0, 0
		for _, v := range pattern {
			barW := float64(v) * p.ModuleWidth
			if barW < 1 {
				barW = 1
			}
			if segIdx == 0 && segCnt >= 3 {
				segIdx++; segCnt = 0
			} else if segIdx == 1 && segCnt >= 16 {
				segIdx++; segCnt = 0
			} else if segIdx == 2 && segCnt >= 5 {
				segIdx++; segCnt = 0
			} else if segIdx == 3 && segCnt >= 16 {
				segIdx++; segCnt = 0
			}
			isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
			if isBar {
				bh := p.NormalBarHeight
				if isTall {
					bh = p.TallBarHeight
				}
				x1 := int(math.Round(currentX))
				x2 := int(math.Round(currentX + barW))
				if x2 > x1 {
					fillRect(img, x1, 0, x2, int(math.Round(bh)), b.foreColor)
				}
			}
			segCnt++
			currentX += barW
			isBar = !isBar
		}
	case "JAN13":
		segIdx, segCnt := 0, 0
		for _, v := range pattern {
			barW := float64(v) * p.ModuleWidth
			if barW < 1 {
				barW = 1
			}
			if segIdx == 0 && segCnt >= 3 {
				segIdx++; segCnt = 0
			} else if segIdx == 1 && segCnt >= 24 {
				segIdx++; segCnt = 0
			} else if segIdx == 2 && segCnt >= 5 {
				segIdx++; segCnt = 0
			} else if segIdx == 3 && segCnt >= 24 {
				segIdx++; segCnt = 0
			}
			isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
			if isBar {
				bh := p.NormalBarHeight
				if isTall {
					bh = p.TallBarHeight
				}
				x1 := int(math.Round(currentX))
				x2 := int(math.Round(currentX + barW))
				if x2 > x1 {
					fillRect(img, x1, 0, x2, int(math.Round(bh)), b.foreColor)
				}
			}
			segCnt++
			currentX += barW
			isBar = !isBar
		}
	case "UPCA":
		segIdx, segCnt := 0, 0
		for _, v := range pattern {
			barW := float64(v) * p.ModuleWidth
			if barW < 1 {
				barW = 1
			}
			if segIdx == 0 && segCnt >= 7 {
				segIdx++; segCnt = 0
			} else if segIdx == 1 && segCnt >= 21 {
				segIdx++; segCnt = 0
			} else if segIdx == 2 && segCnt >= 3 {
				segIdx++; segCnt = 0
			} else if segIdx == 3 && segCnt >= 21 {
				segIdx++; segCnt = 0
			}
			isTall := (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
			if isBar {
				bh := p.NormalBarHeight
				if isTall {
					bh = p.TallBarHeight
				}
				x1 := int(math.Round(currentX))
				x2 := int(math.Round(currentX + barW))
				if x2 > x1 {
					fillRect(img, x1, 0, x2, int(math.Round(bh)), b.foreColor)
				}
			}
			segCnt++
			currentX += barW
			isBar = !isBar
		}
	case "UPCE":
		accUnits := 0
		for _, v := range pattern {
			barW := float64(v) * p.ModuleWidth
			if barW < 1 {
				barW = 1
			}
			isStartGuard := accUnits < 3
			isEndGuard := accUnits >= 45
			isTall := (isStartGuard || isEndGuard) && isBar
			if isBar {
				bh := p.NormalBarHeight
				if isTall {
					bh = p.TallBarHeight
				}
				x1 := int(math.Round(currentX))
				x2 := int(math.Round(currentX + barW))
				if x2 > x1 {
					fillRect(img, x1, 0, x2, int(math.Round(bh)), b.foreColor)
				}
			}
			accUnits += v
			currentX += barW
			isBar = !isBar
		}
	}

	// Draw text using font
	fontSize := p.SVGFontSize
	face := getDefaultFontFace(fontSize)
	if face != nil {
		defer face.Close()
		textY := int(math.Round(p.TextY))

		switch clsName {
		case "JAN8":
			if len(displayCode) >= 8 {
				b.drawJANUPCTextPNG(img, face, displayCode[:4], p.BarcodeStartX+3*p.ModuleWidth, p.BarcodeStartX+31*p.ModuleWidth, p.LeftMargin, textY, 4)
				b.drawJANUPCTextPNG(img, face, displayCode[4:8], p.BarcodeStartX+36*p.ModuleWidth, p.BarcodeStartX+64*p.ModuleWidth, p.LeftMargin, textY, 4)
			}
		case "JAN13":
			if len(displayCode) >= 13 {
				drawCharCenteredPNG(img, face, rune(displayCode[0]), int(math.Round(p.PosA+p.DigitWidth*0.3)), textY, b.foreColor)
				b.drawJANUPCTextPNG(img, face, displayCode[1:7], p.BarcodeStartX+3*p.ModuleWidth, p.BarcodeStartX+45*p.ModuleWidth, p.LeftMargin, textY, 6)
				b.drawJANUPCTextPNG(img, face, displayCode[7:13], p.BarcodeStartX+50*p.ModuleWidth, p.BarcodeStartX+92*p.ModuleWidth, p.LeftMargin, textY, 6)
			}
		case "UPCA":
			if len(displayCode) >= 12 {
				drawCharCenteredPNG(img, face, rune(displayCode[0]), int(math.Round(p.PosA+p.DigitWidth*0.3)), textY, b.foreColor)
				b.drawJANUPCTextPNG(img, face, displayCode[1:6], p.BarcodeStartX+10*p.ModuleWidth, p.BarcodeStartX+31*p.ModuleWidth, p.LeftMargin, textY, 5)
				b.drawJANUPCTextPNG(img, face, displayCode[6:11], p.BarcodeStartX+34*p.ModuleWidth, p.BarcodeStartX+55*p.ModuleWidth, p.LeftMargin, textY, 5)
				endX := p.BarcodeStartX + p.BarcodeWidth
				drawCharCenteredPNG(img, face, rune(displayCode[11]), int(math.Round(endX+p.DigitWidth*0.7)), textY, b.foreColor)
			}
		case "UPCE":
			if len(displayCode) >= 8 {
				drawCharCenteredPNG(img, face, rune(displayCode[0]), int(math.Round(p.PosA+p.DigitWidth*0.3)), textY, b.foreColor)
				startB := p.BarcodeStartX + 3*p.ModuleWidth + p.CharWidth*0.5
				endC := p.BarcodeStartX + 45*p.ModuleWidth - p.CharWidth*0.5
				b.drawJANUPCTextPNG(img, face, displayCode[1:7], startB, endC, p.LeftMargin, textY, 6)
				endX := p.BarcodeStartX + p.BarcodeWidth
				drawCharCenteredPNG(img, face, rune(displayCode[7]), int(math.Round(endX+p.DigitWidth*0.7)), textY, b.foreColor)
			}
		}
	}

	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, width, height)
	}

	return b.encodeImage(img)
}

func (b *BarcodeBase1D) drawJANUPCTextPNG(img *image.NRGBA, face interface{ Close() error }, text string, startPos, endPos, margin float64, textY, count int) {
	// Use font.Face interface
	ff, ok := face.(interface {
		Close() error
		GlyphAdvance(r rune) (advance interface{ Round() int }, ok bool)
	})
	_ = ff
	_ = ok

	// Simple approach: evenly distribute characters
	section := endPos - startPos
	effSection := section - margin*2
	if effSection <= 0 || count <= 0 {
		return
	}

	fontSize := int(math.Round(float64(b.svgHeight) * 0.2 * b.textFontScale * 0.8))
	if fontSize < 8 {
		fontSize = 8
	}

	face2 := getDefaultFontFace(fontSize)
	if face2 == nil {
		return
	}
	defer face2.Close()

	if b.TextEvenSpacing && count > 1 {
		spacing := effSection / float64(count-1)
		for i := 0; i < count && i < len(text); i++ {
			digitX := startPos + margin + float64(i)*spacing
			drawCharCenteredPNG(img, face2, rune(text[i]), int(math.Round(digitX)), textY, b.foreColor)
		}
	} else {
		drawTextCenteredPNG(img, face2, text, int(math.Round(startPos+section/2)), textY, b.foreColor)
	}
}

// Note: fmt is used by error formatting in other functions in this file
var _ = fmt.Errorf
