package barcode

import (
	"strings"
	"testing"
)

func TestJAN8Encode7Digits(t *testing.T) {
	j := NewJAN8(FormatSVG)
	pattern, err := j.Encode("4901234")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	// JAN8: 3 + 4*4 + 5 + 4*4 + 3 = 43 total width units (but run-length varies)
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
}

func TestJAN8Encode8Digits(t *testing.T) {
	j := NewJAN8(FormatSVG)
	// First compute check digit
	cd := CalculateCheckDigitJAN8("4901234")
	code := "4901234" + cd
	pattern, err := j.Encode(code)
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
}

func TestJAN8InvalidCheckDigit(t *testing.T) {
	j := NewJAN8(FormatSVG)
	_, err := j.Encode("49012340") // wrong check digit
	if err == nil {
		t.Fatal("expected error for invalid check digit")
	}
}

func TestJAN8NonNumeric(t *testing.T) {
	j := NewJAN8(FormatSVG)
	_, err := j.Encode("490123A")
	if err == nil {
		t.Fatal("expected error for non-numeric input")
	}
}

func TestJAN8DrawSVG(t *testing.T) {
	j := NewJAN8(FormatSVG)
	err := j.Draw("4901234", 200, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := j.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestJAN8DrawPNG(t *testing.T) {
	j := NewJAN8(FormatPNG)
	err := j.Draw("4901234", 200, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := j.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
}

func TestJAN8CheckDigit(t *testing.T) {
	cd := CalculateCheckDigitJAN8("4901234")
	// 4*3+9*1+0*3+1*1+2*3+3*1+4*3 = 43; (10-43%10)%10 = 7
	if cd != "7" {
		t.Errorf("expected check digit '7', got '%s'", cd)
	}
}
