package barcode

import (
	"fmt"
	"strconv"
)

var (
	jan8LeftPatterns  = []int{0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b}
	jan8RightPatterns = []int{0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74}
)

// CalculateCheckDigitJAN8 computes the JAN8 check digit (Modulus 10, weight 3-1).
func CalculateCheckDigitJAN8(src string) string {
	total := 0
	for i := 0; i < 7; i++ {
		d := int(src[i] - '0')
		if i%2 == 0 {
			total += d * 3
		} else {
			total += d
		}
	}
	cd := (10 - (total % 10)) % 10
	return strconv.Itoa(cd)
}

// JAN8 (EAN-8) encoder.
type JAN8 struct {
	BarcodeBase1D
	ExtendedGuard bool
}

// NewJAN8 creates a new JAN8 encoder.
func NewJAN8(outputFormat string) *JAN8 {
	j := &JAN8{ExtendedGuard: true}
	j.InitBase1D(outputFormat)
	return j
}

// Encode returns the bar/space width pattern for JAN8.
func (j *JAN8) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}
	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("JAN8 barcode requires numeric digits only")
		}
	}

	s := code
	if len(s) == 8 {
		expected := CalculateCheckDigitJAN8(s[:7])
		if string(s[7]) != expected {
			return nil, fmt.Errorf("invalid check digit")
		}
	} else if len(s) == 7 {
		s = s + CalculateCheckDigitJAN8(s)
	} else {
		return nil, fmt.Errorf("JAN8 barcode requires 7 or 8 digits")
	}

	leftHalf := s[:4]
	rightHalf := s[4:8]

	var result []int
	// Start guard (101)
	result = append(result, 1, 1, 1)

	// Left 4 digits (L-pattern)
	for i := 0; i < 4; i++ {
		digit := int(leftHalf[i] - '0')
		pattern := jan8LeftPatterns[digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// Center guard (01010)
	result = append(result, 1, 1, 1, 1, 1)

	// Right 4 digits (R-pattern)
	for i := 0; i < 4; i++ {
		digit := int(rightHalf[i] - '0')
		pattern := jan8RightPatterns[digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// End guard (101)
	result = append(result, 1, 1, 1)

	return result, nil
}

// Draw renders the JAN8 barcode with extended guard bars.
func (j *JAN8) Draw(code string, width, height int) error {
	pattern, err := j.Encode(code)
	if err != nil {
		return err
	}
	// Get display code
	s := code
	if len(s) == 7 {
		s = s + CalculateCheckDigitJAN8(s)
	}

	if j.IsSVGOutput() {
		if j.ShowText && j.ExtendedGuard {
			return j.drawSVGJAN8(pattern, s, width, height)
		}
		display := ""
		if j.ShowText {
			display = s
		}
		return j.drawSVGBars(pattern, width, height, display)
	}
	return j.drawPNGJANUPC(pattern, s, width, height, "JAN8")
}
