package barcode

import (
	"fmt"
	"strconv"
)

var (
	jan13ParityPatterns = []int{0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a}
	jan13LeftPatterns   = [2][]int{
		{0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b}, // L-pattern
		{0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17}, // G-pattern
	}
	jan13RightPatterns = []int{0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74}
)

// CalculateCheckDigitJAN13 computes the JAN13 check digit (Modulus 10, weight 1-3).
func CalculateCheckDigitJAN13(src string) string {
	total := 0
	for i := 0; i < 12; i++ {
		d := int(src[i] - '0')
		if i%2 == 0 {
			total += d
		} else {
			total += d * 3
		}
	}
	cd := (10 - (total % 10)) % 10
	return strconv.Itoa(cd)
}

// JAN13 (EAN-13) encoder.
type JAN13 struct {
	BarcodeBase1D
	ExtendedGuard bool
}

// NewJAN13 creates a new JAN13 encoder.
func NewJAN13(outputFormat string) *JAN13 {
	j := &JAN13{ExtendedGuard: true}
	j.InitBase1D(outputFormat)
	return j
}

// Encode returns the bar/space width pattern for JAN13.
func (j *JAN13) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}
	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("JAN13 barcode requires numeric digits only")
		}
	}

	s := code
	if len(s) == 13 {
		expected := CalculateCheckDigitJAN13(s[:12])
		if string(s[12]) != expected {
			return nil, fmt.Errorf("invalid check digit")
		}
	} else if len(s) == 12 {
		s = s + CalculateCheckDigitJAN13(s)
	} else {
		return nil, fmt.Errorf("JAN13 barcode requires 12 or 13 digits")
	}

	firstDigit := int(s[0] - '0')
	leftHalf := s[1:7]
	rightHalf := s[7:13]

	var result []int
	// Start guard (101)
	result = append(result, 1, 1, 1)

	// Left 6 digits (L or G pattern based on first digit parity)
	chkOE := 0x20
	for i := 0; i < 6; i++ {
		digit := int(leftHalf[i] - '0')
		flgOE := 0
		if (jan13ParityPatterns[firstDigit] & (chkOE >> i)) != 0 {
			flgOE = 1
		}
		pattern := jan13LeftPatterns[flgOE][digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// Center guard (01010)
	result = append(result, 1, 1, 1, 1, 1)

	// Right 6 digits (R-pattern)
	for i := 0; i < 6; i++ {
		digit := int(rightHalf[i] - '0')
		pattern := jan13RightPatterns[digit]
		result = append(result, bitsToRunlength(pattern, 7)...)
	}

	// End guard (101)
	result = append(result, 1, 1, 1)

	return result, nil
}

// Draw renders the JAN13 barcode with extended guard bars.
func (j *JAN13) Draw(code string, width, height int) error {
	pattern, err := j.Encode(code)
	if err != nil {
		return err
	}
	s := code
	if len(s) == 12 {
		s = s + CalculateCheckDigitJAN13(s)
	}

	if j.IsSVGOutput() {
		if j.ShowText && j.ExtendedGuard {
			return j.drawSVGJAN13(pattern, s, width, height)
		}
		display := ""
		if j.ShowText {
			display = s
		}
		return j.drawSVGBars(pattern, width, height, display)
	}
	return j.drawPNGJANUPC(pattern, s, width, height, "JAN13")
}
