package barcode

import (
	"strings"
	"testing"
)

func TestITFEncode(t *testing.T) {
	itf := NewITF(FormatSVG)
	pattern, err := itf.Encode("1234567890")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
	// 10 digits = 5 pairs; start(4) + 5*10 + stop(3) = 57
	if len(pattern) != 57 {
		t.Errorf("expected 57 elements, got %d", len(pattern))
	}
}

func TestITFOddLength(t *testing.T) {
	itf := NewITF(FormatSVG)
	// Odd length should be padded
	pattern, err := itf.Encode("12345")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	// Padded to 012345 = 6 digits = 3 pairs; start(4) + 3*10 + stop(3) = 37
	if len(pattern) != 37 {
		t.Errorf("expected 37 elements, got %d", len(pattern))
	}
}

func TestITFNonNumeric(t *testing.T) {
	itf := NewITF(FormatSVG)
	_, err := itf.Encode("123A")
	if err == nil {
		t.Fatal("expected error for non-numeric input")
	}
}

func TestITFDrawSVG(t *testing.T) {
	itf := NewITF(FormatSVG)
	err := itf.Draw("1234567890", 400, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := itf.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
}
