package barcode

import "fmt"

var itfDigitPatterns = []int{
	0x06, 0x11, 0x09, 0x18, 0x05, 0x14, 0x0c, 0x03, 0x12, 0x0a,
}

// ITF (Interleaved 2 of 5) encoder.
type ITF struct {
	BarcodeBase1D
}

// NewITF creates a new ITF encoder.
func NewITF(outputFormat string) *ITF {
	t := &ITF{}
	t.InitBase1D(outputFormat)
	return t
}

// Encode returns the bar/space width pattern for ITF.
func (t *ITF) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("ITF barcode requires numeric digits only")
		}
	}

	s := code
	if len(s)%2 != 0 {
		s = "0" + s
	}

	wide := 2
	if t.minLineWidth%2 != 0 {
		wide = 3
	}

	var result []int
	// Start pattern
	result = append(result, 1, 1, 1, 1)

	// Encode digit pairs
	for i := 0; i < len(s); i += 2 {
		d1 := int(s[i] - '0')
		d2 := int(s[i+1] - '0')
		p1 := itfDigitPatterns[d1]
		p2 := itfDigitPatterns[d2]

		// Interleave the two 5-bit patterns
		combined := 0x01
		for j := 0; j < 5; j++ {
			bit1 := 0
			if (p1 & (0x10 >> j)) != 0 {
				bit1 = 1
			}
			bit2 := 0
			if (p2 & (0x10 >> j)) != 0 {
				bit2 = 1
			}
			combined = (combined << 1) | bit1
			combined = (combined << 1) | bit2
		}
		combined &= 0x3FF // 10 bits

		for j := 0; j < 10; j++ {
			if (combined & (0x200 >> j)) != 0 {
				result = append(result, wide)
			} else {
				result = append(result, 1)
			}
		}
	}

	// Stop pattern
	result = append(result, wide, 1, 1)

	return result, nil
}

// Draw renders the ITF barcode.
func (t *ITF) Draw(code string, width, height int) error {
	return t.BarcodeBase1D.Draw(code, width, height, t)
}
