package barcode

import (
	"strings"
	"testing"
)

func TestGS1DataBarLimitedEncode(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	pattern, err := g.Encode("0000000000000")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) != 46 {
		t.Errorf("expected 46 elements, got %d", len(pattern))
	}
	if g.GTIN14() == "" {
		t.Error("GTIN14 should not be empty")
	}
}

func TestGS1DataBarLimitedEncodeShort(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	pattern, err := g.Encode("12345678")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) != 46 {
		t.Errorf("expected 46 elements, got %d", len(pattern))
	}
}

func TestGS1DataBarLimitedInvalidLeading(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	_, err := g.Encode("2000000000000")
	if err == nil {
		t.Fatal("expected error for leading digit > 1")
	}
}

func TestGS1DataBarLimitedNonNumeric(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	_, err := g.Encode("123ABC45")
	if err == nil {
		t.Fatal("expected error for non-numeric")
	}
}

func TestGS1DataBarLimitedTooShort(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	_, err := g.Encode("1234567")
	if err == nil {
		t.Fatal("expected error for < 8 digits")
	}
}

func TestGS1DataBarLimitedDrawSVG(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	err := g.Draw("0000000000000", 300, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := g.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBarLimitedDrawPNG(t *testing.T) {
	g := NewGS1DataBarLimited(FormatPNG)
	err := g.Draw("0000000000000", 300, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := g.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
}

func TestGS1DataBarLimitedHumanReadable(t *testing.T) {
	g := NewGS1DataBarLimited(FormatSVG)
	g.Encode("0000000000000")
	hr := g.HumanReadable()
	if !strings.HasPrefix(hr, "(01)") {
		t.Errorf("HumanReadable should start with (01), got '%s'", hr)
	}
}
