package barcode

import "fmt"

// ISO 24724 Limited tables
var leftRegThresholds = []int{0, 183064, 820064, 1000776, 1491021, 1979845, 1996939}
var leftRegOffsets = []int{0, 183064, 820064, 1000776, 1491021, 1979845, 1996939}

var tEvenLtd = []int{28, 728, 6454, 203, 2408, 1, 16632}
var modulesOddLtd = []int{17, 13, 9, 15, 11, 19, 7}
var modulesEvenLtd = []int{9, 13, 17, 11, 15, 7, 19}
var widestOddLtd = []int{6, 5, 3, 5, 4, 8, 1}
var widestEvenLtd = []int{3, 4, 6, 4, 5, 1, 8}

var checksumWeightLtd = []int{
	1, 3, 9, 27, 81, 65, 17, 51, 64, 14,
	42, 37, 22, 66, 20, 60, 2, 6, 18, 54,
	73, 41, 34, 13, 39, 28, 84, 74,
}

var finderPatternLtd = []int{
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 3, 2, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 3, 2, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 2, 1, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 3, 2, 1, 1,
	1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 3, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
	1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 3, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
	1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 3, 1, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 3, 2, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 3, 2, 1, 1,
	1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 2, 1, 1,
	1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
	1, 1, 2, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
	1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 2, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 3, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 1, 2, 1, 2, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 2, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 3, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 2, 2, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 2, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 3, 1, 2, 1, 2, 2, 3, 1, 1, 1,
	1, 1, 1, 1, 3, 1, 2, 2, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 2, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 1, 1, 1, 3, 2, 2, 1, 2, 1, 3, 1, 1, 1,
	1, 2, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
}

func determineGroupLtd(regValue int) int {
	group := 0
	if regValue > 183063 {
		group = 1
	}
	if regValue > 820063 {
		group = 2
	}
	if regValue > 1000775 {
		group = 3
	}
	if regValue > 1491020 {
		group = 4
	}
	if regValue > 1979844 {
		group = 5
	}
	if regValue > 1996938 {
		group = 6
	}
	return group
}

// GS1DataBarLimited encodes GS1 DataBar Limited.
type GS1DataBarLimited struct {
	BarcodeBase1D
	gtin14      string
	totalWidths []int
}

// NewGS1DataBarLimited creates a new GS1 DataBar Limited encoder.
func NewGS1DataBarLimited(outputFormat string) *GS1DataBarLimited {
	g := &GS1DataBarLimited{}
	g.InitBase1D(outputFormat)
	g.ShowText = false
	return g
}

// GTIN14 returns the GTIN-14 with check digit after encoding.
func (g *GS1DataBarLimited) GTIN14() string { return g.gtin14 }

// HumanReadable returns "(01)GTIN-14".
func (g *GS1DataBarLimited) HumanReadable() string { return "(01)" + g.gtin14 }

// Encode encodes GS1 DataBar Limited.
func (g *GS1DataBarLimited) Encode(code string) ([]int, error) {
	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("GS1 DataBar Limited requires numeric digits only")
		}
	}
	if len(code) < 8 || len(code) > 13 {
		return nil, fmt.Errorf("input must be 8-13 digits")
	}

	padded := padLeft(code, 13)

	if padded[0] != '0' && padded[0] != '1' {
		return nil, fmt.Errorf("GS1 DataBar Limited requires leading digit 0 or 1 after padding")
	}

	accum := parseInt64(padded)
	leftReg := int(accum / 2013571)
	rightReg := int(accum % 2013571)

	leftGroup := determineGroupLtd(leftReg)
	rightGroup := determineGroupLtd(rightReg)

	if leftGroup > 0 {
		leftReg -= leftRegOffsets[leftGroup]
	}
	if rightGroup > 0 {
		rightReg -= leftRegOffsets[rightGroup]
	}

	leftOdd := leftReg / tEvenLtd[leftGroup]
	leftEven := leftReg % tEvenLtd[leftGroup]
	rightOdd := rightReg / tEvenLtd[rightGroup]
	rightEven := rightReg % tEvenLtd[rightGroup]

	// Generate width patterns (14 elements each)
	leftWidths := make([]int, 14)
	rightWidths := make([]int, 14)

	w := getWidths(leftOdd, modulesOddLtd[leftGroup], 7, widestOddLtd[leftGroup], 1)
	for i := 0; i < 7; i++ {
		leftWidths[i*2] = w[i]
	}
	w = getWidths(leftEven, modulesEvenLtd[leftGroup], 7, widestEvenLtd[leftGroup], 0)
	for i := 0; i < 7; i++ {
		leftWidths[i*2+1] = w[i]
	}
	w = getWidths(rightOdd, modulesOddLtd[rightGroup], 7, widestOddLtd[rightGroup], 1)
	for i := 0; i < 7; i++ {
		rightWidths[i*2] = w[i]
	}
	w = getWidths(rightEven, modulesEvenLtd[rightGroup], 7, widestEvenLtd[rightGroup], 0)
	for i := 0; i < 7; i++ {
		rightWidths[i*2+1] = w[i]
	}

	// Checksum (modulo 89)
	checksum := 0
	for i := 0; i < 14; i++ {
		checksum += checksumWeightLtd[i] * leftWidths[i]
		checksum += checksumWeightLtd[i+14] * rightWidths[i]
	}
	checksum %= 89

	checkElements := finderPatternLtd[checksum*14 : checksum*14+14]

	// Build total widths (46 elements)
	totalWidths := make([]int, 46)
	totalWidths[0] = 1
	totalWidths[1] = 1
	totalWidths[44] = 1
	totalWidths[45] = 1

	for i := 0; i < 14; i++ {
		totalWidths[i+2] = leftWidths[i]
	}
	for i := 0; i < 14; i++ {
		totalWidths[i+16] = checkElements[i]
	}
	for i := 0; i < 14; i++ {
		totalWidths[i+30] = rightWidths[i]
	}

	cd := calculateGTINCheckDigit(padded)
	g.gtin14 = padded + string(rune('0'+cd))
	g.totalWidths = totalWidths

	return totalWidths, nil
}

// Draw renders the GS1 DataBar Limited barcode.
func (g *GS1DataBarLimited) Draw(code string, width, height int) error {
	pattern, err := g.Encode(code)
	if err != nil {
		return err
	}
	// DataBar Limited is single-row, starts with space
	if g.IsSVGOutput() {
		return g.drawSVGBars(pattern, width, height, "")
	}
	return g.renderBarsToPNG(pattern, width, height, "")
}
