package barcode

import (
	"strings"
	"testing"
)

func TestGS1DataBarExpandedEncodeBasic(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	pattern, err := g.Encode("[01]09501101530003")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
}

func TestGS1DataBarExpandedEncodeWithAI(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	pattern, err := g.Encode("[01]09501101530003[3103]000123")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
	hr := g.HumanReadable()
	if hr == "" {
		t.Error("HumanReadable should not be empty")
	}
}

func TestGS1DataBarExpandedEncodeMethod2(t *testing.T) {
	// Method 2: doesn't start with "01"
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	pattern, err := g.Encode("[10]12345678")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
}

func TestGS1DataBarExpandedDrawSVG(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	err := g.Draw("[01]09501101530003", 500, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := g.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBarExpandedDrawPNG(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatPNG, Unstacked, 2)
	err := g.Draw("[01]09501101530003", 500, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := g.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
}

func TestGS1DataBarExpandedDrawStacked(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, StackedExp, 2)
	err := g.Draw("[01]09501101530003", 300, 150)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, _ := g.GetSVG()
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBarExpandedEmpty(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	_, err := g.Encode("")
	if err == nil {
		t.Fatal("expected error for empty input")
	}
}

func TestGS1DataBarExpandedRowCount(t *testing.T) {
	g := NewGS1DataBarExpanded(FormatSVG, Unstacked, 2)
	g.Encode("[01]09501101530003")
	if g.RowCount() != 1 {
		t.Errorf("Unstacked should have 1 row, got %d", g.RowCount())
	}
}
