package barcode

import (
	"fmt"
	"strings"
)

// ExpandedSymbolType specifies the GS1 DataBar Expanded variant.
type ExpandedSymbolType int

const (
	Unstacked ExpandedSymbolType = 0
	StackedExp ExpandedSymbolType = 1
)

// Encoding modes
const (
	encNumeric    = 0
	encAlpha      = 1
	encISO        = 2
	encInvalid    = 3
	encAnyEnc     = 4
	encAlphaOrISO = 5
)

// ISO 24724 Tables for GS1 DataBar Expanded
var gSumExp = []int{0, 348, 1388, 2948, 3988}
var tEvenExp = []int{4, 20, 52, 104, 204}
var modulesOddExp = []int{12, 10, 8, 6, 4}
var modulesEvenExp = []int{5, 7, 9, 11, 13}
var widestOddExp = []int{7, 5, 4, 3, 1}
var widestEvenExp = []int{2, 4, 5, 6, 8}

var checksumWeightExp = []int{
	1, 3, 9, 27, 81, 32, 96, 77, 20, 60, 180, 118, 143, 7, 21, 63, 189,
	145, 13, 39, 117, 140, 209, 205, 193, 157, 49, 147, 19, 57, 171, 91,
	62, 186, 136, 197, 169, 85, 44, 132, 185, 133, 188, 142, 4, 12, 36,
	108, 113, 128, 173, 97, 80, 29, 87, 50, 150, 28, 84, 41, 123, 158, 52,
	156, 46, 138, 203, 187, 139, 206, 196, 166, 76, 17, 51, 153, 37, 111,
	122, 155, 43, 129, 176, 106, 107, 110, 119, 146, 16, 48, 144, 10, 30,
	90, 59, 177, 109, 116, 137, 200, 178, 112, 125, 164, 70, 210, 208, 202,
	184, 130, 179, 115, 134, 191, 151, 31, 93, 68, 204, 190, 148, 22, 66,
	198, 172, 94, 71, 2, 6, 18, 54, 162, 64, 192, 154, 40, 120, 149, 25,
	75, 14, 42, 126, 167, 79, 26, 78, 23, 69, 207, 199, 175, 103, 98, 83,
	38, 114, 131, 182, 124, 161, 61, 183, 127, 170, 88, 53, 159, 55, 165,
	73, 8, 24, 72, 5, 15, 45, 135, 194, 160, 58, 174, 100, 89,
}

var finderPatternExp = []int{
	1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1,
	1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5,
	6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2,
}

var finderSequence = []int{
	1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6,
	3, 8, 0, 0, 0, 0, 0, 0, 0, 1, 10, 3, 8, 5, 0, 0, 0, 0, 0, 0, 1, 10, 3,
	8, 7, 12, 0, 0, 0, 0, 0, 1, 10, 3, 8, 9, 12, 11, 0, 0, 0, 0, 1, 2, 3,
	4, 5, 6, 7, 8, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 10, 9, 0, 0, 1, 2, 3, 4,
	5, 6, 7, 10, 11, 12, 0, 1, 2, 3, 4, 5, 8, 7, 10, 9, 12, 11,
}

var weightRows = []int{
	0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6,
	3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 10, 3, 4,
	13, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13,
	14, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
	11, 12, 21, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
	15, 16, 21, 22, 19, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7,
	8, 9, 10, 11, 12, 13, 14, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
	9, 10, 11, 12, 17, 18, 15, 16, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
	9, 10, 11, 12, 17, 18, 19, 20, 21, 22, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
	13, 14, 11, 12, 17, 18, 15, 16, 21, 22, 19, 20,
}

// ISO/IEC 646 special character encoding table
var isoChars = map[byte]string{
	'!': "11101000", '"': "11101001", '%': "11101010",
	'&': "11101011", '\'': "11101100", '(': "11101101",
	')': "11101110", '*': "11101111", '+': "11110000",
	',': "11110001", '-': "11110010", '.': "11110011",
	'/': "11110100", ':': "11110101", ';': "11110110",
	'<': "11110111", '=': "11111000", '>': "11111001",
	'?': "11111010", '_': "11111011", ' ': "11111100",
}

// GS1DataBarExpanded encodes GS1 DataBar Expanded.
type GS1DataBarExpanded struct {
	BarcodeBase1D
	symbolType    ExpandedSymbolType
	numColumns    int
	humanReadable string
	elements      []int
	patterns      []string
	rowHeights    []int
	rowCount      int
	// internal
	binaryString     string
	generalField     string
	generalFieldType []int
	encMethodResult  int
}

// NewGS1DataBarExpanded creates a new GS1 DataBar Expanded encoder.
func NewGS1DataBarExpanded(outputFormat string, symbolType ExpandedSymbolType, numColumns int) *GS1DataBarExpanded {
	g := &GS1DataBarExpanded{
		symbolType: symbolType,
		numColumns: numColumns,
	}
	g.InitBase1D(outputFormat)
	g.ShowText = false
	return g
}

// SetSymbolType sets the symbol type.
func (g *GS1DataBarExpanded) SetSymbolType(st ExpandedSymbolType) { g.symbolType = st }

// SetNumColumns sets the number of columns for stacked mode.
func (g *GS1DataBarExpanded) SetNumColumns(n int) { g.numColumns = n }

// Patterns returns run-length encoded pattern strings for each row.
func (g *GS1DataBarExpanded) Patterns() []string { return g.patterns }

// RowHeights returns row heights in X units.
func (g *GS1DataBarExpanded) RowHeights() []int { return g.rowHeights }

// RowCount returns the number of rows.
func (g *GS1DataBarExpanded) RowCount() int { return g.rowCount }

// HumanReadable returns the human-readable text.
func (g *GS1DataBarExpanded) HumanReadable() string { return g.humanReadable }

func intToBits(value, numBits int) string {
	result := make([]byte, numBits)
	for j := 0; j < numBits; j++ {
		if value&(1<<(numBits-1-j)) != 0 {
			result[j] = '1'
		} else {
			result[j] = '0'
		}
	}
	return string(result)
}

func calculateRemainder(binaryStringLength int) int {
	remainder := 12 - (binaryStringLength % 12)
	if remainder == 12 {
		remainder = 0
	}
	if binaryStringLength < 36 {
		remainder = 36 - binaryStringLength
	}
	return remainder
}

func bin2patFromBlack(binary string) string {
	if len(binary) == 0 {
		return ""
	}
	pattern := bin2pat(binary)
	if binary[0] == '0' {
		pattern = "0" + pattern
	}
	return pattern
}

// applyGeneralFieldRules applies ISO 24724 general field encoding rules.
// Returns true if last block is numeric and odd size.
func (g *GS1DataBarExpanded) applyGeneralFieldRules() bool {
	gft := g.generalFieldType
	n := len(g.generalField)
	if n == 0 {
		return false
	}

	// Build blocks
	blockLen := []int{1}
	blockType := []int{gft[0]}
	for i := 1; i < n; i++ {
		if gft[i] == gft[i-1] {
			blockLen[len(blockLen)-1]++
		} else {
			blockLen = append(blockLen, 1)
			blockType = append(blockType, gft[i])
		}
	}
	blockCount := len(blockLen)

	// Apply rules
	for i := 0; i < blockCount; i++ {
		cur := blockType[i]
		nxt := -1
		if i < blockCount-1 {
			nxt = blockType[i+1]
		}

		if cur == encISO && i != blockCount-1 {
			if nxt == encAnyEnc && blockLen[i+1] >= 4 {
				blockType[i+1] = encNumeric
			}
			if nxt == encAnyEnc && blockLen[i+1] < 4 {
				blockType[i+1] = encISO
			}
			if nxt == encAlphaOrISO && blockLen[i+1] >= 5 {
				blockType[i+1] = encAlpha
			}
			if nxt == encAlphaOrISO && blockLen[i+1] < 5 {
				blockType[i+1] = encISO
			}
		}

		if cur == encAlphaOrISO {
			blockType[i] = encAlpha
		}

		if cur == encAlpha && i != blockCount-1 {
			if nxt == encAnyEnc && blockLen[i+1] >= 6 {
				blockType[i+1] = encNumeric
			}
			if nxt == encAnyEnc && blockLen[i+1] < 6 {
				if i == blockCount-2 && blockLen[i+1] >= 4 {
					blockType[i+1] = encNumeric
				} else {
					blockType[i+1] = encAlpha
				}
			}
		}

		if cur == encAnyEnc {
			blockType[i] = encNumeric
		}
	}

	// Merge adjacent same-type blocks
	if blockCount > 1 {
		i := 1
		for i < len(blockLen) {
			if blockType[i-1] == blockType[i] {
				blockLen[i-1] += blockLen[i]
				blockLen = append(blockLen[:i], blockLen[i+1:]...)
				blockType = append(blockType[:i], blockType[i+1:]...)
				if i > 1 {
					i--
				}
			} else {
				i++
			}
		}
		blockCount = len(blockLen)
	}

	// Odd-size numeric blocks (not the last)
	for i := 0; i < blockCount-1; i++ {
		if blockType[i] == encNumeric && (blockLen[i]&1) != 0 {
			blockLen[i]--
			blockLen[i+1]++
		}
	}

	// Expand back to per-character types
	j := 0
	for i := 0; i < blockCount; i++ {
		for k := 0; k < blockLen[i]; k++ {
			g.generalFieldType[j] = blockType[i]
			j++
		}
	}

	return blockType[blockCount-1] == encNumeric && (blockLen[blockCount-1]&1) != 0
}

// calculateBinaryString builds the binary data string.
func (g *GS1DataBarExpanded) calculateBinaryString(source, originalCode string) error {
	aiTag := "{AI}"
	lastMode := encNumeric

	// Determine encoding method
	encodingMethod := 2
	if len(source) >= 16 && source[0] == '0' && source[1] == '1' {
		encodingMethod = 1
		g.humanReadable = "(01)" + originalCode[2:]
		for {
			p := strings.Index(g.humanReadable, aiTag)
			if p < 0 {
				break
			}
			aiStart := p + len(aiTag)
			g.humanReadable = g.humanReadable[:p] + "(" +
				g.humanReadable[aiStart:aiStart+2] + ")" +
				g.humanReadable[aiStart+2:]
		}
	} else {
		encodingMethod = 2
		g.humanReadable = originalCode
		for {
			p := strings.Index(g.humanReadable, aiTag)
			if p < 0 {
				break
			}
			aiStart := p + len(aiTag)
			g.humanReadable = g.humanReadable[:p] + "(" +
				g.humanReadable[aiStart:aiStart+2] + ")" +
				g.humanReadable[aiStart+2:]
		}
	}

	// Check for compressed methods > 2
	if len(source) >= 20 && encodingMethod == 1 &&
		source[2] == '9' && source[16] == '3' {
		g.checkCompressedMethods(source, encodingMethod)
		encodingMethod = g.encMethodResult
	}

	// Method header and read position
	readPosn := g.applyMethodHeader(encodingMethod, len(source))

	// Verify compressed data is numeric
	for i := 0; i < readPosn; i++ {
		c := source[i]
		if !(c >= '0' && c <= '9') && c != '[' && c != ']' {
			return fmt.Errorf("invalid characters in input data")
		}
	}

	// Encode compressed data field
	g.encodeCompressed(encodingMethod, source)

	// General purpose data field
	g.generalField = source[readPosn:]
	g.generalFieldType = make([]int, len(g.generalField))
	for i := range g.generalFieldType {
		g.generalFieldType[i] = encISO
	}

	if len(g.generalField) > 0 {
		lastMode = g.encodeGeneralField(lastMode)
	}

	// Length check
	if len(g.binaryString) > 252 {
		return fmt.Errorf("input too long")
	}

	// Padding
	remainder := calculateRemainder(len(g.binaryString))
	padstring := ""
	padI := remainder
	if len(g.generalField) > 0 && lastMode == encNumeric {
		padstring = "0000"
		padI = remainder - 4
	}
	for padI > 0 {
		padstring += "00100"
		padI -= 5
	}
	if remainder > 0 && remainder <= len(padstring) {
		g.binaryString += padstring[:remainder]
	} else if remainder > 0 {
		g.binaryString += padstring
	}

	// Patch variable length symbol bit field
	bs := g.binaryString
	dataCharCount := len(bs)/12 + 1
	patch0 := "0"
	if dataCharCount&1 != 0 {
		patch0 = "1"
	}
	patch1 := "0"
	if len(bs) > 156 {
		patch1 = "1"
	}
	patch := patch0 + patch1

	switch encodingMethod {
	case 1:
		g.binaryString = bs[:2] + patch + bs[4:]
	case 2:
		g.binaryString = bs[:3] + patch + bs[5:]
	case 5, 6:
		g.binaryString = bs[:6] + patch + bs[8:]
	}

	return nil
}

func (g *GS1DataBarExpanded) checkCompressedMethods(source string, baseMethod int) {
	g.encMethodResult = baseMethod

	if len(source) >= 26 && source[17] == '1' && source[18] == '0' {
		weight := 0.0
		for i := 0; i < 6; i++ {
			weight = weight*10 + float64(source[20+i]-'0')
		}
		if weight < 99999.0 {
			if source[19] == '3' && len(source) == 26 {
				if weight/1000.0 <= 32.767 {
					g.encMethodResult = 3
					g.humanReadable = "(01)" + source[2:16] + "(3103)" + source[20:]
				}
			}
			if len(source) == 34 {
				g.setWeightDateMethod(source, weight, "310", []int{7, 9, 11, 13})
			}
		}
	}

	if len(source) >= 26 && source[17] == '2' && source[18] == '0' {
		weight := 0.0
		for i := 0; i < 6; i++ {
			weight = weight*10 + float64(source[20+i]-'0')
		}
		if weight < 99999.0 {
			if (source[19] == '2' || source[19] == '3') && len(source) == 26 {
				if source[19] == '3' {
					if weight/1000.0 <= 22.767 {
						g.encMethodResult = 4
						g.humanReadable = "(01)" + source[2:16] + "(3203)" + source[20:]
					}
				} else {
					if weight/100.0 <= 99.99 {
						g.encMethodResult = 4
						g.humanReadable = "(01)" + source[2:16] + "(3202)" + source[20:]
					}
				}
			}
			if len(source) == 34 {
				g.setWeightDateMethod(source, weight, "320", []int{8, 10, 12, 14})
			}
		}
	}

	if source[17] == '9' {
		if source[18] == '2' && source[19] >= '0' && source[19] <= '3' {
			g.encMethodResult = 5
			g.humanReadable = "(01)" + source[2:16] +
				"(392" + string(source[19]) + ")" + source[20:]
		}
		if source[18] == '3' && source[19] >= '0' && source[19] <= '3' {
			g.encMethodResult = 6
			g.humanReadable = "(01)" + source[2:16] +
				"(393" + string(source[19]) + ")" + source[20:]
		}
	}
}

func (g *GS1DataBarExpanded) setWeightDateMethod(source string, weight float64, aiPrefix string, methods []int) {
	if len(source) != 34 {
		return
	}
	dateAIMap := map[byte]int{'1': methods[0], '3': methods[1], '5': methods[2], '7': methods[3]}
	if source[26] == '1' {
		if m, ok := dateAIMap[source[27]]; ok {
			dateAIs := map[int]string{methods[0]: "11", methods[1]: "13", methods[2]: "15", methods[3]: "17"}
			g.encMethodResult = m
			g.humanReadable = "(01)" + source[2:16] +
				"(" + aiPrefix + string(source[19]) + ")" + source[20:26] +
				"(" + dateAIs[m] + ")" + source[28:]
		}
	}
}

func (g *GS1DataBarExpanded) applyMethodHeader(encodingMethod, sourceLen int) int {
	type headerInfo struct {
		header string
		rp     int
	}
	headers := map[int]headerInfo{
		1: {"1XX", 16}, 2: {"00XX", 0},
		3: {"0100", -1}, 4: {"0101", -1},
		5: {"01100XX", 20}, 6: {"01101XX", 23},
		7: {"0111000", -1}, 8: {"0111001", -1},
		9: {"0111010", -1}, 10: {"0111011", -1},
		11: {"0111100", -1}, 12: {"0111101", -1},
		13: {"0111110", -1}, 14: {"0111111", -1},
	}
	h := headers[encodingMethod]
	g.binaryString += h.header
	if h.rp < 0 {
		return sourceLen
	}
	return h.rp
}

func (g *GS1DataBarExpanded) encodeCompressed(encodingMethod int, source string) {
	switch {
	case encodingMethod == 1:
		groupVal := int(source[2] - '0')
		g.binaryString += intToBits(groupVal, 4)
		for i := 1; i <= 4; i++ {
			gv := 100*int(source[i*3]-'0') + 10*int(source[i*3+1]-'0') + int(source[i*3+2]-'0')
			g.binaryString += intToBits(gv, 10)
		}

	case encodingMethod == 3 || encodingMethod == 4:
		for i := 1; i <= 4; i++ {
			gv := 100*int(source[i*3]-'0') + 10*int(source[i*3+1]-'0') + int(source[i*3+2]-'0')
			g.binaryString += intToBits(gv, 10)
		}
		gv := 0
		for i := 0; i < 6; i++ {
			gv = gv*10 + int(source[20+i]-'0')
		}
		if encodingMethod == 4 && source[19] == '3' {
			gv += 10000
		}
		g.binaryString += intToBits(gv, 15)

	case encodingMethod >= 7 && encodingMethod <= 14:
		for i := 1; i <= 4; i++ {
			gv := 100*int(source[i*3]-'0') + 10*int(source[i*3+1]-'0') + int(source[i*3+2]-'0')
			g.binaryString += intToBits(gv, 10)
		}
		gv := int(source[19] - '0')
		for i := 0; i < 5; i++ {
			gv = gv*10 + int(source[21+i]-'0')
		}
		g.binaryString += intToBits(gv, 20)
		if len(source) == 34 {
			gv = (((10*int(source[28]-'0') + int(source[29]-'0')) * 384) +
				((10*int(source[30]-'0')+int(source[31]-'0'))-1)*32 +
				10*int(source[32]-'0') + int(source[33]-'0'))
		} else {
			gv = 38400
		}
		g.binaryString += intToBits(gv, 16)

	case encodingMethod == 5 || encodingMethod == 6:
		for i := 1; i <= 4; i++ {
			gv := 100*int(source[i*3]-'0') + 10*int(source[i*3+1]-'0') + int(source[i*3+2]-'0')
			g.binaryString += intToBits(gv, 10)
		}
		decMap := map[byte]string{'0': "00", '1': "01", '2': "10", '3': "11"}
		g.binaryString += decMap[source[19]]
		if encodingMethod == 6 {
			gv := 0
			for i := 0; i < 3; i++ {
				gv = gv*10 + int(source[20+i]-'0')
			}
			g.binaryString += intToBits(gv, 10)
		}
	}
}

func (g *GS1DataBarExpanded) encodeGeneralField(lastMode int) int {
	gf := g.generalField
	gft := g.generalFieldType
	n := len(gf)

	// Classify each character
	hasInvalid := false
	for i := 0; i < n; i++ {
		c := gf[i]
		if c < ' ' || c > 'z' {
			gft[i] = encInvalid
			hasInvalid = true
		} else {
			gft[i] = encISO
		}
		if c == '#' || c == '$' || c == '@' || c == '\\' || c == '^' || c == '`' {
			gft[i] = encInvalid
			hasInvalid = true
		}
		if (c >= 'A' && c <= 'Z') || c == '*' || c == ',' || c == '-' || c == '.' || c == '/' {
			gft[i] = encAlphaOrISO
		}
		if (c >= '0' && c <= '9') || c == '[' {
			gft[i] = encAnyEnc
		}
	}

	if hasInvalid {
		return lastMode // caller should check for errors
	}

	// Propagate ISOIEC/ALPHA_OR_ISO to following FNC1
	for i := 0; i < n-1; i++ {
		if gft[i] == encISO && gf[i+1] == '[' {
			gft[i+1] = encISO
		}
	}
	for i := 0; i < n-1; i++ {
		if gft[i] == encAlphaOrISO && gf[i+1] == '[' {
			gft[i+1] = encAlphaOrISO
		}
	}

	// Apply rules
	latch := g.applyGeneralFieldRules()

	// Set initial mode
	if gft[0] == encAlpha {
		g.binaryString += "0000"
		lastMode = encAlpha
	}
	if gft[0] == encISO {
		g.binaryString += "0000"
		g.binaryString += "00100"
		lastMode = encISO
	}

	// Encode characters
	i := 0
	for {
		mode := gft[i]

		if mode == encNumeric {
			if lastMode != encNumeric {
				g.binaryString += "000"
			}
			d1 := 10
			if gf[i] != '[' {
				d1 = int(gf[i] - '0')
			}
			d2 := 10
			if gf[i+1] != '[' {
				d2 = int(gf[i+1] - '0')
			}
			value := 11*d1 + d2 + 8
			g.binaryString += intToBits(value, 7)
			i += 2
			lastMode = encNumeric

		} else if mode == encAlpha {
			if i != 0 {
				if lastMode == encNumeric {
					g.binaryString += "0000"
				}
				if lastMode == encISO {
					g.binaryString += "00100"
				}
			}
			c := gf[i]
			if c >= '0' && c <= '9' {
				g.binaryString += intToBits(int(c)-43, 5)
			}
			if c >= 'A' && c <= 'Z' {
				g.binaryString += intToBits(int(c)-33, 6)
			}
			lastMode = encAlpha
			if c == '[' {
				g.binaryString += "01111"
				lastMode = encNumeric
			}
			if c == '*' {
				g.binaryString += "111010"
			}
			if c == ',' {
				g.binaryString += "111011"
			}
			if c == '-' {
				g.binaryString += "111100"
			}
			if c == '.' {
				g.binaryString += "111101"
			}
			if c == '/' {
				g.binaryString += "111110"
			}
			i++

		} else if mode == encISO {
			if i != 0 {
				if lastMode == encNumeric {
					g.binaryString += "0000"
					g.binaryString += "00100"
				}
				if lastMode == encAlpha {
					g.binaryString += "00100"
				}
			}
			c := gf[i]
			if c >= '0' && c <= '9' {
				g.binaryString += intToBits(int(c)-43, 5)
			}
			if c >= 'A' && c <= 'Z' {
				g.binaryString += intToBits(int(c)-1, 7)
			}
			if c >= 'a' && c <= 'z' {
				g.binaryString += intToBits(int(c)-7, 7)
			}
			lastMode = encISO
			if c == '[' {
				g.binaryString += "01111"
				lastMode = encNumeric
			}
			if v, ok := isoChars[c]; ok {
				g.binaryString += v
			}
			i++
		} else {
			i++
		}

		currentLength := i
		if latch {
			currentLength++
		}
		if currentLength >= n {
			break
		}
	}

	// Handle single remaining digit (latch case)
	remainder := calculateRemainder(len(g.binaryString))
	if latch {
		if lastMode == encNumeric {
			if remainder >= 4 && remainder <= 6 {
				value := int(gf[i]-'0') + 1
				g.binaryString += intToBits(value, 4)
			} else {
				d1 := int(gf[i] - '0')
				d2 := 10
				value := 11*d1 + d2 + 8
				g.binaryString += intToBits(value, 7)
			}
		} else {
			value := int(gf[i]) - 43
			g.binaryString += intToBits(value, 5)
		}
	}

	return lastMode
}

// Encode encodes GS1 DataBar Expanded.
func (g *GS1DataBarExpanded) Encode(code string) ([]int, error) {
	content := strings.ReplaceAll(code, "{AI}", "")
	if len(content) == 0 {
		return nil, fmt.Errorf("input data is empty")
	}

	g.binaryString = "0" // Linkage flag
	if err := g.calculateBinaryString(content, code); err != nil {
		return nil, err
	}

	// Data characters from binary string
	dataChars := len(g.binaryString) / 12

	// Calculate vs values (12-bit segments)
	vs := make([]int, dataChars)
	for i := 0; i < dataChars; i++ {
		v := 0
		for j := 0; j < 12; j++ {
			if g.binaryString[i*12+j] == '1' {
				v += (2048 >> j)
			}
		}
		vs[i] = v
	}

	// Determine groups and calculate widths
	group := make([]int, dataChars)
	charWidths := make([][8]int, dataChars)

	for i := 0; i < dataChars; i++ {
		if vs[i] <= 347 {
			group[i] = 1
		} else if vs[i] <= 1387 {
			group[i] = 2
		} else if vs[i] <= 2947 {
			group[i] = 3
		} else if vs[i] <= 3987 {
			group[i] = 4
		} else {
			group[i] = 5
		}

		gi := group[i] - 1
		vo := (vs[i] - gSumExp[gi]) / tEvenExp[gi]
		ve := (vs[i] - gSumExp[gi]) % tEvenExp[gi]

		w := getWidths(vo, modulesOddExp[gi], 4, widestOddExp[gi], 0)
		charWidths[i][0] = w[0]
		charWidths[i][2] = w[1]
		charWidths[i][4] = w[2]
		charWidths[i][6] = w[3]

		w = getWidths(ve, modulesEvenExp[gi], 4, widestEvenExp[gi], 1)
		charWidths[i][1] = w[0]
		charWidths[i][3] = w[1]
		charWidths[i][5] = w[2]
		charWidths[i][7] = w[3]
	}

	// Checksum (modulo 211)
	checksum := 0
	for i := 0; i < dataChars; i++ {
		row := weightRows[(((dataChars-2)/2)*21)+i]
		for j := 0; j < 8; j++ {
			checksum += charWidths[i][j] * checksumWeightExp[row*8+j]
		}
	}

	checkChar := 211*((dataChars+1)-4) + (checksum % 211)

	// Check character widths
	var cg int
	if checkChar <= 347 {
		cg = 1
	} else if checkChar <= 1387 {
		cg = 2
	} else if checkChar <= 2947 {
		cg = 3
	} else if checkChar <= 3987 {
		cg = 4
	} else {
		cg = 5
	}

	cOdd := (checkChar - gSumExp[cg-1]) / tEvenExp[cg-1]
	cEven := (checkChar - gSumExp[cg-1]) % tEvenExp[cg-1]

	checkWidths := [8]int{}
	w := getWidths(cOdd, modulesOddExp[cg-1], 4, widestOddExp[cg-1], 0)
	checkWidths[0], checkWidths[2] = w[0], w[1]
	checkWidths[4], checkWidths[6] = w[2], w[3]
	w = getWidths(cEven, modulesEvenExp[cg-1], 4, widestEvenExp[cg-1], 1)
	checkWidths[1], checkWidths[3] = w[0], w[1]
	checkWidths[5], checkWidths[7] = w[2], w[3]

	// Build elements array
	patternWidth := ((((dataChars+1)/2)+((dataChars+1)&1))*5 +
		(dataChars+1)*8 + 4)
	elements := make([]int, patternWidth)
	elements[0] = 1
	elements[1] = 1
	elements[patternWidth-2] = 1
	elements[patternWidth-1] = 1

	// Finder patterns
	finderCount := (dataChars+1)/2 + ((dataChars + 1) & 1)
	for i := 0; i < finderCount; i++ {
		k := ((((dataChars+1)-2)/2 + ((dataChars + 1) & 1)) - 1) * 11
		k += i
		for j := 0; j < 5; j++ {
			elements[21*i+j+10] = finderPatternExp[(finderSequence[k]-1)*5+j]
		}
	}

	// Check character at position 2
	for i := 0; i < 8; i++ {
		elements[i+2] = checkWidths[i]
	}

	// Odd-index data characters (forward)
	for i := 1; i < dataChars; i += 2 {
		for j := 0; j < 8; j++ {
			elements[((i-1)/2)*21+23+j] = charWidths[i][j]
		}
	}

	// Even-index data characters (reversed)
	for i := 0; i < dataChars; i += 2 {
		for j := 0; j < 8; j++ {
			elements[(i/2)*21+15+j] = charWidths[i][7-j]
		}
	}

	g.elements = elements

	// Generate patterns for rendering
	g.generatePatterns(elements, patternWidth, dataChars)

	return elements, nil
}

func (g *GS1DataBarExpanded) generatePatterns(elements []int, patternWidth, dataChars int) {
	if g.symbolType == Unstacked {
		g.rowCount = 1
		g.rowHeights = []int{-1}
		pattern := "0"
		for i := 0; i < patternWidth; i++ {
			pattern += string(rune(elements[i] + '0'))
		}
		g.patterns = []string{pattern}
	} else {
		g.generateStacked(elements, patternWidth, dataChars)
	}
}

func (g *GS1DataBarExpanded) generateStacked(elements []int, patternWidth, dataChars int) {
	codeblocks := (dataChars+1)/2 + ((dataChars + 1) % 2)
	blocksPerRow := g.numColumns
	if blocksPerRow < 1 || blocksPerRow > 10 {
		blocksPerRow = 2
	}

	stackRows := codeblocks / blocksPerRow
	if codeblocks%blocksPerRow > 0 {
		stackRows++
	}

	rowCount := stackRows*4 - 3
	g.rowHeights = make([]int, rowCount)
	g.patterns = make([]string, rowCount)
	symbolRow := 0
	currentBlock := 0

	for currentRow := 1; currentRow <= stackRows; currentRow++ {
		subElements := make([]int, 235)
		specialCaseRow := false

		subElements[0] = 1
		subElements[1] = 1
		elementsInSub := 2

		reader := 0
		leftToRight := false

		for {
			if (blocksPerRow&1) != 0 ||
				(currentRow&1) != 0 ||
				(currentRow == stackRows &&
					codeblocks != currentRow*blocksPerRow &&
					(((currentRow*blocksPerRow)-codeblocks)&1) != 0) {
				leftToRight = true
				baseI := 2 + currentBlock*21
				for j := 0; j < 21; j++ {
					if baseI+j < patternWidth {
						subElements[j+reader*21+2] = elements[baseI+j]
						elementsInSub++
					}
				}
			} else {
				leftToRight = false
				if currentRow*blocksPerRow < codeblocks {
					baseI := 2 + ((currentRow*blocksPerRow)-reader-1)*21
					for j := 0; j < 21; j++ {
						if baseI+j < patternWidth {
							subElements[(20-j)+reader*21+2] = elements[baseI+j]
							elementsInSub++
						}
					}
				} else {
					kOff := (currentRow * blocksPerRow) - codeblocks
					lOff := (currentRow * blocksPerRow) - reader - 1
					baseI := 2 + (lOff-kOff)*21
					for j := 0; j < 21; j++ {
						if baseI+j < patternWidth {
							subElements[(20-j)+reader*21+2] = elements[baseI+j]
							elementsInSub++
						}
					}
				}
			}

			reader++
			currentBlock++
			if reader >= blocksPerRow || currentBlock >= codeblocks {
				break
			}
		}

		// Row stop
		subElements[elementsInSub] = 1
		subElements[elementsInSub+1] = 1
		elementsInSub += 2

		// Build pattern
		g.rowHeights[symbolRow] = -1
		black := true

		if (currentRow & 1) != 0 {
			g.patterns[symbolRow] = "0"
			black = false
		} else {
			if currentRow == stackRows &&
				codeblocks != currentRow*blocksPerRow &&
				(((currentRow*blocksPerRow)-codeblocks)&1) != 0 {
				specialCaseRow = true
				subElements[0] = 2
				g.patterns[symbolRow] = "0"
				black = false
			} else {
				g.patterns[symbolRow] = ""
			}
		}

		// Build separator binary and pattern string
		writer := 0
		sepBin := ""
		for i := 0; i < elementsInSub; i++ {
			g.patterns[symbolRow] += string(rune(subElements[i] + '0'))
			for k := 0; k < subElements[i]; k++ {
				if black {
					sepBin += "0"
				} else {
					sepBin += "1"
				}
			}
			black = !black
			writer += subElements[i]
		}

		// Fix separator binary
		if writer > 4 {
			sepBin = "0000" + sepBin[4:]
			if len(sepBin) > writer {
				sepBin = sepBin[:writer]
			}
		}

		for j := 0; j < reader; j++ {
			k := 49*j + 18
			if specialCaseRow {
				k = 49*j + 19
			}
			if leftToRight {
				for ii := 0; ii < 15; ii++ {
					idx := ii + k
					if idx-1 >= 0 && idx < len(sepBin) &&
						sepBin[idx-1] == '1' && sepBin[idx] == '1' {
						sepBin = sepBin[:idx] + "0" + sepBin[idx+1:]
					}
				}
			} else {
				for ii := 14; ii >= 0; ii-- {
					idx := ii + k
					if idx >= 0 && idx+1 < len(sepBin) &&
						sepBin[idx+1] == '1' && sepBin[idx] == '1' {
						sepBin = sepBin[:idx] + "0" + sepBin[idx+1:]
					}
				}
			}
		}

		sepPattern := bin2patFromBlack(sepBin)

		// Separator rows
		if currentRow != 1 {
			midSep := "05"
			for cnt := 5; cnt < 49*blocksPerRow; cnt += 2 {
				midSep += "11"
			}
			g.patterns[symbolRow-2] = midSep
			g.rowHeights[symbolRow-2] = 1
			g.patterns[symbolRow-1] = sepPattern
			g.rowHeights[symbolRow-1] = 1
		}

		if currentRow != stackRows {
			g.patterns[symbolRow+1] = sepPattern
			g.rowHeights[symbolRow+1] = 1
		}

		symbolRow += 4
	}

	g.rowCount = rowCount
}

// Draw renders the GS1 DataBar Expanded barcode.
func (g *GS1DataBarExpanded) Draw(code string, width, height int) error {
	_, err := g.Encode(code)
	if err != nil {
		return err
	}
	if g.IsSVGOutput() {
		return drawMultiRowSVG(&g.BarcodeBase, g.patterns, g.rowHeights, width, height)
	}
	return drawMultiRowPNG(&g.BarcodeBase, g.patterns, g.rowHeights, width, height)
}
