package barcode

import (
	"strings"
	"testing"
)

func TestGS1DataBar14EncodeOmni(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	pattern, err := g.Encode("0000000000000")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) != 46 {
		t.Errorf("expected 46 elements, got %d", len(pattern))
	}
	if g.GTIN14() == "" {
		t.Error("GTIN14 should not be empty")
	}
	if len(g.GTIN14()) != 14 {
		t.Errorf("expected 14-digit GTIN, got %d", len(g.GTIN14()))
	}
}

func TestGS1DataBar14EncodeShort(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	pattern, err := g.Encode("1234")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) != 46 {
		t.Errorf("expected 46 elements, got %d", len(pattern))
	}
}

func TestGS1DataBar14NonNumeric(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	_, err := g.Encode("123ABC")
	if err == nil {
		t.Fatal("expected error for non-numeric")
	}
}

func TestGS1DataBar14DrawSVGOmni(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	err := g.Draw("0000000000000", 300, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := g.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBar14DrawPNGOmni(t *testing.T) {
	g := NewGS1DataBar14(FormatPNG, Omnidirectional)
	err := g.Draw("0000000000000", 300, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := g.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
}

func TestGS1DataBar14DrawSVGStacked(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Stacked)
	err := g.Draw("0000000000000", 200, 50)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, _ := g.GetSVG()
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBar14DrawSVGStackedOmni(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, StackedOmnidirectional)
	err := g.Draw("0000000000000", 200, 80)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, _ := g.GetSVG()
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG missing <svg tag")
	}
}

func TestGS1DataBar14RowCount(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	g.Encode("0000000000000")
	if g.RowCount() != 1 {
		t.Errorf("Omni should have 1 row, got %d", g.RowCount())
	}

	g2 := NewGS1DataBar14(FormatSVG, Stacked)
	g2.Encode("0000000000000")
	if g2.RowCount() != 3 {
		t.Errorf("Stacked should have 3 rows, got %d", g2.RowCount())
	}

	g3 := NewGS1DataBar14(FormatSVG, StackedOmnidirectional)
	g3.Encode("0000000000000")
	if g3.RowCount() != 5 {
		t.Errorf("StackedOmni should have 5 rows, got %d", g3.RowCount())
	}
}

func TestGS1DataBar14HumanReadable(t *testing.T) {
	g := NewGS1DataBar14(FormatSVG, Omnidirectional)
	g.Encode("0000000000000")
	hr := g.HumanReadable()
	if !strings.HasPrefix(hr, "(01)") {
		t.Errorf("HumanReadable should start with (01), got '%s'", hr)
	}
}
