package barcode

import "fmt"

// SymbolType14 specifies the GS1 DataBar 14 variant.
type SymbolType14 int

const (
	Omnidirectional       SymbolType14 = 0
	Stacked               SymbolType14 = 1
	StackedOmnidirectional SymbolType14 = 2
)

// ISO 24724 Table 3, 4
var gSumTable14 = []int{0, 161, 961, 2015, 2715, 0, 336, 1036, 1516}
var tTable14 = []int{1, 10, 34, 70, 126, 4, 20, 48, 81}
var modulesOdd14 = []int{12, 10, 8, 6, 4, 5, 7, 9, 11}
var modulesEven14 = []int{4, 6, 8, 10, 12, 10, 8, 6, 4}
var widestOdd14 = []int{8, 6, 4, 3, 1, 2, 4, 6, 8}
var widestEven14 = []int{1, 3, 5, 6, 8, 7, 5, 3, 1}

var checksumWeight14 = []int{
	1, 3, 9, 27, 2, 6, 18, 54,
	4, 12, 36, 29, 8, 24, 72, 58,
	16, 48, 65, 37, 32, 17, 51, 74,
	64, 34, 23, 69, 49, 68, 46, 59,
}

var finderPattern14 = []int{
	3, 8, 2, 1, 1,
	3, 5, 5, 1, 1,
	3, 3, 7, 1, 1,
	3, 1, 9, 1, 1,
	2, 7, 4, 1, 1,
	2, 5, 6, 1, 1,
	2, 3, 8, 1, 1,
	1, 5, 7, 1, 1,
	1, 3, 9, 1, 1,
}

// GS1DataBar14 encodes GS1 DataBar 14 (RSS-14).
type GS1DataBar14 struct {
	BarcodeBase1D
	symbolType  SymbolType14
	gtin14      string
	totalWidths []int
	patterns    []string
	rowHeights  []int
	rowCount    int
}

// NewGS1DataBar14 creates a new GS1 DataBar 14 encoder.
func NewGS1DataBar14(outputFormat string, symbolType SymbolType14) *GS1DataBar14 {
	g := &GS1DataBar14{symbolType: symbolType}
	g.InitBase1D(outputFormat)
	g.ShowText = false
	return g
}

// SetSymbolType sets the symbol type.
func (g *GS1DataBar14) SetSymbolType(st SymbolType14) { g.symbolType = st }

// SymbolType returns the current symbol type.
func (g *GS1DataBar14) SymbolType() SymbolType14 { return g.symbolType }

// GTIN14 returns the GTIN-14 with check digit after encoding.
func (g *GS1DataBar14) GTIN14() string { return g.gtin14 }

// Patterns returns run-length encoded pattern strings for each row.
func (g *GS1DataBar14) Patterns() []string { return g.patterns }

// RowHeights returns row heights in X units (-1 means variable).
func (g *GS1DataBar14) RowHeights() []int { return g.rowHeights }

// RowCount returns the number of rows.
func (g *GS1DataBar14) RowCount() int { return g.rowCount }

// HumanReadable returns "(01)GTIN-14".
func (g *GS1DataBar14) HumanReadable() string { return "(01)" + g.gtin14 }

// Encode encodes GS1 DataBar 14.
func (g *GS1DataBar14) Encode(code string) ([]int, error) {
	for _, c := range code {
		if c < '0' || c > '9' {
			return nil, fmt.Errorf("GS1 DataBar requires numeric digits only")
		}
	}
	if len(code) > 13 {
		return nil, fmt.Errorf("input too long (max 13 digits)")
	}

	// Left-pad to 13 digits
	padded := padLeft(code, 13)

	// Convert to numeric value
	accum := parseInt64(padded)

	// Split into left_reg and right_reg
	leftReg := int(accum / 4537077)
	rightReg := int(accum % 4537077)

	// 4 data characters
	dataChars := [4]int{
		leftReg / 1597,
		leftReg % 1597,
		rightReg / 1597,
		rightReg % 1597,
	}

	// Data groups
	dataGroup := [4]int{}
	for _, idx := range []int{0, 2} {
		dataGroup[idx] = 0
		if dataChars[idx] >= 161 {
			dataGroup[idx] = 1
		}
		if dataChars[idx] >= 961 {
			dataGroup[idx] = 2
		}
		if dataChars[idx] >= 2015 {
			dataGroup[idx] = 3
		}
		if dataChars[idx] >= 2715 {
			dataGroup[idx] = 4
		}
	}
	for _, idx := range []int{1, 3} {
		dataGroup[idx] = 5
		if dataChars[idx] >= 336 {
			dataGroup[idx] = 6
		}
		if dataChars[idx] >= 1036 {
			dataGroup[idx] = 7
		}
		if dataChars[idx] >= 1516 {
			dataGroup[idx] = 8
		}
	}

	// Calculate odd/even values
	vOdd := [4]int{}
	vEven := [4]int{}
	vOdd[0] = (dataChars[0] - gSumTable14[dataGroup[0]]) / tTable14[dataGroup[0]]
	vEven[0] = (dataChars[0] - gSumTable14[dataGroup[0]]) % tTable14[dataGroup[0]]
	vOdd[1] = (dataChars[1] - gSumTable14[dataGroup[1]]) % tTable14[dataGroup[1]]
	vEven[1] = (dataChars[1] - gSumTable14[dataGroup[1]]) / tTable14[dataGroup[1]]
	vOdd[3] = (dataChars[3] - gSumTable14[dataGroup[3]]) % tTable14[dataGroup[3]]
	vEven[3] = (dataChars[3] - gSumTable14[dataGroup[3]]) / tTable14[dataGroup[3]]
	vOdd[2] = (dataChars[2] - gSumTable14[dataGroup[2]]) / tTable14[dataGroup[2]]
	vEven[2] = (dataChars[2] - gSumTable14[dataGroup[2]]) % tTable14[dataGroup[2]]

	// Width patterns
	dataWidths := [8][4]int{}
	for i := 0; i < 4; i++ {
		if i == 0 || i == 2 {
			w := getWidths(vOdd[i], modulesOdd14[dataGroup[i]], 4, widestOdd14[dataGroup[i]], 1)
			dataWidths[0][i] = w[0]
			dataWidths[2][i] = w[1]
			dataWidths[4][i] = w[2]
			dataWidths[6][i] = w[3]
			w = getWidths(vEven[i], modulesEven14[dataGroup[i]], 4, widestEven14[dataGroup[i]], 0)
			dataWidths[1][i] = w[0]
			dataWidths[3][i] = w[1]
			dataWidths[5][i] = w[2]
			dataWidths[7][i] = w[3]
		} else {
			w := getWidths(vOdd[i], modulesOdd14[dataGroup[i]], 4, widestOdd14[dataGroup[i]], 0)
			dataWidths[0][i] = w[0]
			dataWidths[2][i] = w[1]
			dataWidths[4][i] = w[2]
			dataWidths[6][i] = w[3]
			w = getWidths(vEven[i], modulesEven14[dataGroup[i]], 4, widestEven14[dataGroup[i]], 1)
			dataWidths[1][i] = w[0]
			dataWidths[3][i] = w[1]
			dataWidths[5][i] = w[2]
			dataWidths[7][i] = w[3]
		}
	}

	// Checksum (modulo 79)
	checksum := 0
	for i := 0; i < 8; i++ {
		checksum += checksumWeight14[i] * dataWidths[i][0]
		checksum += checksumWeight14[i+8] * dataWidths[i][1]
		checksum += checksumWeight14[i+16] * dataWidths[i][2]
		checksum += checksumWeight14[i+24] * dataWidths[i][3]
	}
	checksum %= 79

	// Finder pattern selection
	if checksum >= 8 {
		checksum++
	}
	if checksum >= 72 {
		checksum++
	}
	cLeft := checksum / 9
	cRight := checksum % 9

	// Build total widths (46 elements)
	totalWidths := make([]int, 46)
	totalWidths[0] = 1
	totalWidths[1] = 1
	totalWidths[44] = 1
	totalWidths[45] = 1

	for i := 0; i < 8; i++ {
		totalWidths[i+2] = dataWidths[i][0]
		totalWidths[i+15] = dataWidths[7-i][1]
		totalWidths[i+23] = dataWidths[i][3]
		totalWidths[i+36] = dataWidths[7-i][2]
	}

	for i := 0; i < 5; i++ {
		totalWidths[i+10] = finderPattern14[i+5*cLeft]
		totalWidths[i+31] = finderPattern14[(4-i)+5*cRight]
	}

	// GTIN-14 check digit
	cd := calculateGTINCheckDigit(padded)
	g.gtin14 = padded + string(rune('0'+cd))
	g.totalWidths = totalWidths

	// Generate grid and patterns
	g.generateGrid(totalWidths)

	return totalWidths, nil
}

func (g *GS1DataBar14) generateGrid(totalWidths []int) {
	grid := [5][100]bool{}
	rowCount := 0
	symbolWidth := 0

	switch g.symbolType {
	case Omnidirectional:
		writer := 0
		latch := false
		for i := 0; i < 46; i++ {
			for j := 0; j < totalWidths[i]; j++ {
				if latch {
					grid[rowCount][writer] = true
				}
				writer++
			}
			latch = !latch
		}
		if writer > symbolWidth {
			symbolWidth = writer
		}
		rowCount++

	case Stacked:
		// Upper row (row 0)
		writer := 0
		latch := false
		for i := 0; i < 23; i++ {
			for j := 0; j < totalWidths[i]; j++ {
				grid[rowCount][writer] = latch
				writer++
			}
			latch = !latch
		}
		grid[rowCount][writer] = true
		grid[rowCount][writer+1] = false

		// Lower row (row 2)
		rowCount += 2
		grid[rowCount][0] = true
		grid[rowCount][1] = false
		writer = 0
		latch = true
		for i := 23; i < 46; i++ {
			for j := 0; j < totalWidths[i]; j++ {
				grid[rowCount][writer+2] = latch
				writer++
			}
			latch = !latch
		}

		// Separator (row 1)
		for i := 4; i < 46; i++ {
			if grid[rowCount-2][i] == grid[rowCount][i] {
				if !grid[rowCount-2][i] {
					grid[rowCount-1][i] = true
				}
			} else {
				if !grid[rowCount-1][i-1] {
					grid[rowCount-1][i] = true
				}
			}
		}

		rowCount++
		if 50 > symbolWidth {
			symbolWidth = 50
		}

	case StackedOmnidirectional:
		// Upper row (row 0)
		writer := 0
		latch := false
		for i := 0; i < 23; i++ {
			for j := 0; j < totalWidths[i]; j++ {
				grid[rowCount][writer] = latch
				writer++
			}
			latch = !latch
		}
		grid[rowCount][writer] = true
		grid[rowCount][writer+1] = false

		// Lower row (row 4)
		rowCount += 4
		grid[rowCount][0] = true
		grid[rowCount][1] = false
		writer = 0
		latch = true
		for i := 23; i < 46; i++ {
			for j := 0; j < totalWidths[i]; j++ {
				grid[rowCount][writer+2] = latch
				writer++
			}
			latch = !latch
		}

		// Middle separator (row 2)
		for i := 5; i < 46; i += 2 {
			grid[rowCount-2][i] = true
		}

		// Upper separator (row 1)
		for i := 4; i < 46; i++ {
			if !grid[rowCount-4][i] {
				grid[rowCount-3][i] = true
			}
		}
		latchVal := true
		for i := 17; i < 33; i++ {
			if !grid[rowCount-4][i] {
				if latchVal {
					grid[rowCount-3][i] = true
					latchVal = false
				} else {
					grid[rowCount-3][i] = false
					latchVal = true
				}
			} else {
				grid[rowCount-3][i] = false
				latchVal = true
			}
		}

		// Lower separator (row 3)
		for i := 4; i < 46; i++ {
			if !grid[rowCount][i] {
				grid[rowCount-1][i] = true
			}
		}
		latchVal = true
		for i := 16; i < 32; i++ {
			if !grid[rowCount][i] {
				if latchVal {
					grid[rowCount-1][i] = true
					latchVal = false
				} else {
					grid[rowCount-1][i] = false
					latchVal = true
				}
			} else {
				grid[rowCount-1][i] = false
				latchVal = true
			}
		}

		if 50 > symbolWidth {
			symbolWidth = 50
		}
		rowCount++
	}

	// Convert grid to pattern strings
	g.patterns = nil
	g.rowHeights = nil

	for i := 0; i < rowCount; i++ {
		binary := ""
		for j := 0; j < symbolWidth; j++ {
			if grid[i][j] {
				binary += "1"
			} else {
				binary += "0"
			}
		}

		pattern := bin2pat(binary)
		if len(binary) > 0 && binary[0] == '0' {
			pattern = "0" + pattern
		}

		g.patterns = append(g.patterns, pattern)
	}

	// Row heights
	switch g.symbolType {
	case Omnidirectional:
		g.rowHeights = []int{-1}
	case Stacked:
		g.rowHeights = []int{5, 1, 7}
	case StackedOmnidirectional:
		g.rowHeights = []int{-1, 1, 1, 1, -1}
	}

	g.rowCount = rowCount
}

// Draw renders the GS1 DataBar 14 barcode.
func (g *GS1DataBar14) Draw(code string, width, height int) error {
	_, err := g.Encode(code)
	if err != nil {
		return err
	}
	if g.IsSVGOutput() {
		return drawMultiRowSVG(&g.BarcodeBase, g.patterns, g.rowHeights, width, height)
	}
	return drawMultiRowPNG(&g.BarcodeBase, g.patterns, g.rowHeights, width, height)
}
