package barcode

import (
	"fmt"
	"strings"
)

var (
	gs1128Patterns = code128Patterns // same as Code128
	gs1128Start    = []string{"211412", "211214", "211232"}
	gs1128Stop     = "2331112"

	gs1128CodeA    string
	gs1128CodeB    string
	gs1128CodeC    []string
	gs1128OnlyB    = "`abcdefghijklmnopqrstuvwxyz{|}~"
	gs1128EscChars string
)

func init() {
	// CODE-A: space(32) to _(95) + control chars (0-31)
	var a strings.Builder
	for i := 32; i <= 95; i++ {
		a.WriteByte(byte(i))
	}
	for i := 0; i < 32; i++ {
		a.WriteByte(byte(i))
	}
	gs1128CodeA = a.String()

	// CODE-B: space(32) to ~(126) + DEL(127)
	var b strings.Builder
	for i := 32; i <= 126; i++ {
		b.WriteByte(byte(i))
	}
	b.WriteByte(0x7f)
	gs1128CodeB = b.String()

	// CODE-C: "00" to "99"
	gs1128CodeC = make([]string, 100)
	for i := 0; i < 100; i++ {
		gs1128CodeC[i] = fmt.Sprintf("%02d", i)
	}

	// ESC: control chars (0-31)
	var e strings.Builder
	for i := 0; i < 32; i++ {
		e.WriteByte(byte(i))
	}
	gs1128EscChars = e.String()
}

func gs1128IsInt4(s string, pos int) bool {
	if pos+4 > len(s) {
		return false
	}
	for i := pos; i < pos+4; i++ {
		if s[i] < '0' || s[i] > '9' {
			return false
		}
	}
	return true
}

func gs1128GetABC(code string) int {
	for _, c := range code {
		if c < 32 {
			return 0 // CODE-A
		}
		if strings.ContainsRune(gs1128OnlyB, c) {
			return 1 // CODE-B
		}
	}
	return 1 // default B
}

func gs1128SafeSub(s string, pos, length int) string {
	if pos >= len(s) {
		return ""
	}
	end := pos + length
	if end > len(s) {
		end = len(s)
	}
	return s[pos:end]
}

// GS1128 encoder.
type GS1128 struct {
	BarcodeBase1D
}

// NewGS1128 creates a new GS1-128 encoder.
func NewGS1128(outputFormat string) *GS1128 {
	g := &GS1128{}
	g.InitBase1D(outputFormat)
	g.fitWidth = true
	return g
}

// Encode returns the bar/space width pattern for GS1-128.
func (g *GS1128) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	swABC := g.determineStartSet(code)

	var allPattern []int

	// Start code
	ptn := gs1128Start[swABC]
	for _, ch := range ptn {
		allPattern = append(allPattern, int(ch-'0'))
	}

	// Data encoding
	swABC2 := swABC
	i := 0
	for i < len(code) {
		chgFlg := false
		if swABC2 == 0 { // CODE-A
			if gs1128IsInt4(code, i) {
				swABC2 = 2
				chgFlg = true
			}
			if i < len(code) && strings.ContainsRune(gs1128OnlyB, rune(code[i])) {
				swABC2 = 1
				chgFlg = true
			}
		} else if swABC2 == 1 { // CODE-B
			if gs1128IsInt4(code, i) {
				swABC2 = 2
				chgFlg = true
			}
			if i < len(code) && strings.ContainsRune(gs1128EscChars, rune(code[i])) {
				swABC2 = 0
				chgFlg = true
			}
		} else { // CODE-C
			c2 := gs1128SafeSub(code, i, 2)
			if len(c2) < 2 {
				swABC2 = gs1128GetABC(code[i:])
				chgFlg = true
			} else if c2 != "{F" && c2 != "{A" {
				found := false
				for _, cc := range gs1128CodeC {
					if c2 == cc {
						found = true
						break
					}
				}
				if !found {
					swABC2 = gs1128GetABC(code[i:])
					chgFlg = true
				}
			}
		}

		if chgFlg {
			swCode := 99
			if swABC2 == 0 {
				swCode = 101
			} else if swABC2 == 1 {
				swCode = 100
			}
			ptn := gs1128Patterns[swCode]
			for _, ch := range ptn {
				allPattern = append(allPattern, int(ch-'0'))
			}
		}

		// Encode character
		var idx int
		if i+6 <= len(code) && code[i:i+6] == "{FNC1}" {
			idx = 102
			i += 6
		} else if i+4 <= len(code) && code[i:i+4] == "{AI}" {
			i += 4
			continue
		} else if swABC2 == 0 { // CODE-A
			pos := strings.IndexByte(gs1128CodeA, code[i])
			if pos >= 0 {
				idx = pos
			}
			i++
		} else if swABC2 == 1 { // CODE-B
			pos := strings.IndexByte(gs1128CodeB, code[i])
			if pos >= 0 {
				idx = pos
			}
			i++
		} else { // CODE-C
			c2 := gs1128SafeSub(code, i, 2)
			if len(c2) < 2 {
				pos := strings.IndexByte(gs1128CodeB, code[i])
				if pos >= 0 {
					idx = pos
				}
				i++
			} else {
				for ci, cc := range gs1128CodeC {
					if c2 == cc {
						idx = ci
						break
					}
				}
				i += 2
			}
		}

		ptn := gs1128Patterns[idx]
		for _, ch := range ptn {
			allPattern = append(allPattern, int(ch-'0'))
		}
	}

	// Check digit
	cdIdx := g.calcCheckDigit(code)
	ptn = gs1128Patterns[cdIdx]
	for _, ch := range ptn {
		allPattern = append(allPattern, int(ch-'0'))
	}

	// Stop code
	for _, ch := range gs1128Stop {
		allPattern = append(allPattern, int(ch-'0'))
	}

	return allPattern, nil
}

func (g *GS1128) determineStartSet(code string) int {
	cntNum, cntF, cntAI := 0, 0, 0

	for cntNum+cntF+cntAI < len(code)-1 {
		pos := cntNum + cntF + cntAI
		if pos+6 <= len(code) && code[pos:pos+6] == "{FNC1}" {
			cntF += 6
			continue
		}
		if pos+4 <= len(code) && code[pos:pos+4] == "{AI}" {
			cntAI += 4
			continue
		}
		if pos >= len(code) {
			break
		}
		if code[pos] < '0' || code[pos] > '9' {
			break
		}
		cntNum++
	}

	if cntNum < 3 {
		return gs1128GetABC(code)
	}
	return 2 // CODE-C
}

func (g *GS1128) calcCheckDigit(code string) int {
	cntNum, cntF, cntAI := 0, 0, 0

	for cntNum+cntF+cntAI < len(code)-1 {
		pos := cntNum + cntF + cntAI
		if pos+6 <= len(code) && code[pos:pos+6] == "{FNC1}" {
			cntF += 6
			continue
		}
		if pos+4 <= len(code) && code[pos:pos+4] == "{AI}" {
			cntAI += 4
			continue
		}
		if pos >= len(code) {
			break
		}
		if code[pos] < '0' || code[pos] > '9' {
			break
		}
		cntNum++
	}

	swABC := 0
	charSum := 0
	if cntNum < 3 {
		swABC = gs1128GetABC(code)
		if swABC == 1 {
			charSum = 104
		} else {
			charSum = 103
		}
	} else {
		swABC = 2
		charSum = 105
	}

	cnt := 0
	i := 0
	for i < len(code) {
		if swABC == 0 { // CODE-A
			if gs1128IsInt4(code, i) {
				swABC = 2
				cnt++
				charSum += 99 * cnt
			}
			if i < len(code) && strings.ContainsRune(gs1128OnlyB, rune(code[i])) {
				swABC = 1
				cnt++
				charSum += 100 * cnt
			}
		} else if swABC == 1 { // CODE-B
			if gs1128IsInt4(code, i) {
				swABC = 2
				cnt++
				charSum += 99 * cnt
			}
			if i < len(code) && strings.ContainsRune(gs1128EscChars, rune(code[i])) {
				swABC = 0
				cnt++
				charSum += 101 * cnt
			}
		} else { // CODE-C
			c2 := gs1128SafeSub(code, i, 2)
			if len(c2) < 2 {
				swABC = gs1128GetABC(code[i:])
				cnt++
				if swABC == 1 {
					charSum += 100 * cnt
				} else {
					charSum += 101 * cnt
				}
			} else if c2 != "{F" && c2 != "{A" {
				found := false
				for _, cc := range gs1128CodeC {
					if c2 == cc {
						found = true
						break
					}
				}
				if !found {
					swABC = gs1128GetABC(code[i:])
					cnt++
					if swABC == 1 {
						charSum += 100 * cnt
					} else {
						charSum += 101 * cnt
					}
				}
			}
		}

		idx := -1
		if i+6 <= len(code) && code[i:i+6] == "{FNC1}" {
			idx = 102
			i += 6
		} else if i+4 <= len(code) && code[i:i+4] == "{AI}" {
			idx = -1
			i += 4
		} else if swABC == 0 { // CODE-A
			c := gs1128SafeSub(code, i, 1)
			if c != "" {
				pos := strings.IndexByte(gs1128CodeA, c[0])
				if pos >= 0 {
					idx = pos
				}
			}
			i++
		} else if swABC == 1 { // CODE-B
			c := gs1128SafeSub(code, i, 1)
			if c != "" {
				pos := strings.IndexByte(gs1128CodeB, c[0])
				if pos >= 0 {
					idx = pos
				}
			}
			i++
		} else { // CODE-C
			c2 := gs1128SafeSub(code, i, 2)
			if len(c2) < 2 {
				pos := strings.IndexByte(gs1128CodeB, code[i])
				if pos >= 0 {
					idx = pos
				}
				i++
			} else {
				for ci, cc := range gs1128CodeC {
					if c2 == cc {
						idx = ci
						break
					}
				}
				i += 2
			}
		}

		if idx != -1 {
			cnt++
			charSum += idx * cnt
		}
	}

	return charSum % 103
}

// CalcCheckDigitMod10W3 calculates Modulus 10 Weight 3-1 check digit.
func CalcCheckDigitMod10W3(digits string) int {
	total := 0
	weight3 := true
	for i := len(digits) - 1; i >= 0; i-- {
		d := int(digits[i] - '0')
		if weight3 {
			total += d * 3
		} else {
			total += d
		}
		weight3 = !weight3
	}
	mod := total % 10
	if mod == 0 {
		return 0
	}
	return 10 - mod
}

// Draw renders the GS1-128 barcode.
func (g *GS1128) Draw(code string, width, height int) error {
	pattern, err := g.Encode(code)
	if err != nil {
		return err
	}
	display := ""
	if g.ShowText {
		display = strings.ReplaceAll(code, "{FNC1}", "")
		display = strings.ReplaceAll(display, "{AI}", "")
	}
	if g.IsSVGOutput() {
		return g.drawSVGBars(pattern, width, height, display)
	}
	return g.renderBarsToPNG(pattern, width, height, display)
}
