package barcode

import (
	"strings"
	"testing"
)

func TestDataMatrixEncodeBasic(t *testing.T) {
	dm := NewDataMatrix(FormatSVG)
	patt, err := dm.GetPattern("Hello")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}
	if len(patt) == 0 {
		t.Fatal("pattern is empty")
	}
	// "Hello" is 5 bytes, which fits into a 10x10 symbol (3 data words capacity
	// after encoding: each ASCII char = 1 codeword, so 5 codewords -> needs 12x12 or bigger)
	cols := len(patt)
	rows := len(patt[0])
	if cols < 10 || rows < 10 {
		t.Errorf("expected at least 10x10 symbol, got %dx%d", cols, rows)
	}
	// DataMatrix is always square for square sizes, so cols == rows
	if cols != rows {
		t.Errorf("expected square symbol, got %dx%d", cols, rows)
	}

	// Verify finder patterns: bottom-left corner module should be ON (L-shape finder)
	// Row 0 in pattern corresponds to top row visually; the bottom row is rows-1
	// The bottom-left module at [0][rows-1] should be set (finder pattern)
	if !patt[0][rows-1] {
		t.Error("bottom-left corner module should be ON (finder pattern)")
	}
}

func TestDataMatrixDrawSVG(t *testing.T) {
	dm := NewDataMatrix(FormatSVG)
	err := dm.Draw("Test123", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := dm.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
	if !strings.Contains(svg, "</svg>") {
		t.Error("SVG output missing </svg> tag")
	}
	if !strings.Contains(svg, "<rect") {
		t.Error("SVG output missing <rect elements")
	}
}

func TestDataMatrixDrawPNG(t *testing.T) {
	dm := NewDataMatrix(FormatPNG)
	err := dm.Draw("Test123", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := dm.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
	// Check PNG magic bytes
	if data[0] != 0x89 || data[1] != 0x50 {
		t.Error("output is not valid PNG")
	}
}

func TestDataMatrixAutoSize(t *testing.T) {
	dm := NewDataMatrix(FormatSVG)
	dm.SetCodeSize(DxSzAuto)

	// Short data -> small symbol
	patt1, err := dm.GetPattern("AB")
	if err != nil {
		t.Fatalf("GetPattern for short data failed: %v", err)
	}
	size1 := len(patt1)

	// Longer data -> larger symbol
	patt2, err := dm.GetPattern("ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890")
	if err != nil {
		t.Fatalf("GetPattern for long data failed: %v", err)
	}
	size2 := len(patt2)

	if size2 <= size1 {
		t.Errorf("longer data should produce larger symbol: short=%d, long=%d", size1, size2)
	}
}

func TestDataMatrixFixedSize(t *testing.T) {
	dm := NewDataMatrix(FormatSVG)

	// Force 14x14 symbol
	dm.SetCodeSize(DxSz14x14)
	patt, err := dm.GetPattern("AB")
	if err != nil {
		t.Fatalf("GetPattern with fixed size failed: %v", err)
	}
	cols := len(patt)
	rows := len(patt[0])
	if cols != 14 || rows != 14 {
		t.Errorf("expected 14x14, got %dx%d", cols, rows)
	}

	// Force 20x20 symbol
	dm.SetCodeSize(DxSz20x20)
	patt2, err := dm.GetPattern("AB")
	if err != nil {
		t.Fatalf("GetPattern with 20x20 failed: %v", err)
	}
	cols2 := len(patt2)
	rows2 := len(patt2[0])
	if cols2 != 20 || rows2 != 20 {
		t.Errorf("expected 20x20, got %dx%d", cols2, rows2)
	}
}

func TestDataMatrixGetPattern(t *testing.T) {
	dm := NewDataMatrix(FormatPNG)
	patt, err := dm.GetPattern("12345")
	if err != nil {
		t.Fatalf("GetPattern failed: %v", err)
	}

	cols := len(patt)
	rows := len(patt[0])

	// Verify the pattern is reasonable
	if cols < 10 || rows < 10 {
		t.Errorf("pattern too small: %dx%d", cols, rows)
	}

	// Count set modules - should have a mix of ON and OFF
	onCount := 0
	offCount := 0
	for c := 0; c < cols; c++ {
		for r := 0; r < rows; r++ {
			if patt[c][r] {
				onCount++
			} else {
				offCount++
			}
		}
	}
	total := cols * rows
	if onCount == 0 {
		t.Error("no modules are ON")
	}
	if offCount == 0 {
		t.Error("no modules are OFF")
	}
	// DataMatrix typically has roughly 40-60% modules set
	onRatio := float64(onCount) / float64(total)
	if onRatio < 0.2 || onRatio > 0.8 {
		t.Errorf("unusual ON ratio: %.2f (%d/%d)", onRatio, onCount, total)
	}
}

func TestDataMatrixEmpty(t *testing.T) {
	dm := NewDataMatrix(FormatSVG)
	_, err := dm.GetPattern("")
	if err == nil {
		t.Fatal("expected error for empty input")
	}

	err = dm.Draw("", 200)
	if err == nil {
		t.Fatal("expected error for empty input in Draw")
	}
}

func TestDataMatrixDeterministic(t *testing.T) {
	// Encoding the same data twice should produce identical patterns
	dm := NewDataMatrix(FormatSVG)
	patt1, err := dm.GetPattern("Deterministic")
	if err != nil {
		t.Fatalf("first GetPattern failed: %v", err)
	}
	patt2, err := dm.GetPattern("Deterministic")
	if err != nil {
		t.Fatalf("second GetPattern failed: %v", err)
	}

	if len(patt1) != len(patt2) {
		t.Fatalf("pattern col count differs: %d vs %d", len(patt1), len(patt2))
	}
	for c := 0; c < len(patt1); c++ {
		if len(patt1[c]) != len(patt2[c]) {
			t.Fatalf("pattern row count differs at col %d: %d vs %d", c, len(patt1[c]), len(patt2[c]))
		}
		for r := 0; r < len(patt1[c]); r++ {
			if patt1[c][r] != patt2[c][r] {
				t.Errorf("module at [%d][%d] differs between runs", c, r)
			}
		}
	}
}

func TestDataMatrixSchemes(t *testing.T) {
	// Test different encoding schemes produce valid output
	schemes := []struct {
		name   string
		scheme int
		data   string
	}{
		{"ASCII", DxSchemeAscii, "Hello123"},
		{"C40", DxSchemeC40, "HELLO WORLD"},
		{"Text", DxSchemeText, "hello world"},
		{"Base256", DxSchemeBase256, "Hello"},
	}

	for _, tc := range schemes {
		t.Run(tc.name, func(t *testing.T) {
			dm := NewDataMatrix(FormatSVG)
			dm.SetEncodeScheme(tc.scheme)
			patt, err := dm.GetPattern(tc.data)
			if err != nil {
				t.Fatalf("GetPattern with %s scheme failed: %v", tc.name, err)
			}
			if len(patt) == 0 {
				t.Fatalf("empty pattern with %s scheme", tc.name)
			}
		})
	}
}

func TestDataMatrixSizeConstants(t *testing.T) {
	// Verify size constants map to correct symbol sizes
	tests := []struct {
		sizeConst int
		expected  int // expected symbol cols
	}{
		{DxSz10x10, 10},
		{DxSz12x12, 12},
		{DxSz14x14, 14},
		{DxSz16x16, 16},
		{DxSz18x18, 18},
		{DxSz20x20, 20},
		{DxSz22x22, 22},
		{DxSz24x24, 24},
		{DxSz26x26, 26},
		{DxSz32x32, 32},
	}

	for _, tc := range tests {
		cols := dxGetSymbolAttribute(dxAttrSymbolCols, tc.sizeConst)
		if cols != tc.expected {
			t.Errorf("size index %d: expected cols=%d, got %d", tc.sizeConst, tc.expected, cols)
		}
	}
}

func TestDataMatrixRectangular(t *testing.T) {
	// Test rectangular symbol sizes
	dm := NewDataMatrix(FormatSVG)
	dm.SetCodeSize(DxSz8x18)
	patt, err := dm.GetPattern("AB")
	if err != nil {
		t.Fatalf("GetPattern with rectangular size failed: %v", err)
	}
	cols := len(patt)
	rows := len(patt[0])
	if cols != 18 || rows != 8 {
		t.Errorf("expected 18x8 (cols x rows), got %dx%d", cols, rows)
	}
}

func TestDataMatrixImageBase64(t *testing.T) {
	dm := NewDataMatrix(FormatPNG)
	err := dm.Draw("Base64Test", 200)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	b64, err := dm.GetImageBase64()
	if err != nil {
		t.Fatalf("GetImageBase64 failed: %v", err)
	}
	if !strings.HasPrefix(b64, "data:image/png;base64,") {
		t.Error("base64 output missing data URI prefix")
	}
}
