package barcode

import (
	"fmt"
	"image"
	"image/draw"
	"math"
)

// DataMatrix code size constants.
const (
	DxSzRectAuto  = -3
	DxSzAuto      = -2
	DxSzShapeAuto = -1
	DxSz10x10     = 0
	DxSz12x12     = 1
	DxSz14x14     = 2
	DxSz16x16     = 3
	DxSz18x18     = 4
	DxSz20x20     = 5
	DxSz22x22     = 6
	DxSz24x24     = 7
	DxSz26x26     = 8
	DxSz32x32     = 9
	DxSz36x36     = 10
	DxSz40x40     = 11
	DxSz44x44     = 12
	DxSz48x48     = 13
	DxSz52x52     = 14
	DxSz64x64     = 15
	DxSz72x72     = 16
	DxSz80x80     = 17
	DxSz88x88     = 18
	DxSz96x96     = 19
	DxSz104x104   = 20
	DxSz120x120   = 21
	DxSz132x132   = 22
	DxSz144x144   = 23
	DxSz8x18      = 24
	DxSz8x32      = 25
	DxSz12x26     = 26
	DxSz12x36     = 27
	DxSz16x36     = 28
	DxSz16x48     = 29
)

// DataMatrix encoding scheme constants.
const (
	DxSchemeAutoFast = -2
	DxSchemeAutoBest = -1
	DxSchemeAscii    = 0
	DxSchemeC40      = 1
	DxSchemeText     = 2
	DxSchemeX12      = 3
	DxSchemeEdifact  = 4
	DxSchemeBase256  = 5
	DxSchemeAsciiGS1 = 6
)

// Internal symbol attribute indices.
const (
	dxAttrSymbolRows         = 0
	dxAttrSymbolCols         = 1
	dxAttrDataRegionRows     = 2
	dxAttrDataRegionCols     = 3
	dxAttrHorizDataRegions   = 4
	dxAttrVertDataRegions    = 5
	dxAttrMappingMatrixRows  = 6
	dxAttrMappingMatrixCols  = 7
	dxAttrInterleavedBlocks  = 8
	dxAttrBlockErrorWords    = 9
	dxAttrBlockMaxCorrectable = 10
	dxAttrSymbolDataWords    = 11
	dxAttrSymbolErrorWords   = 12
	dxAttrSymbolMaxCorrectable = 13
)

// Internal mask bits.
const (
	dxMaskBit1 = 0x80
	dxMaskBit2 = 0x40
	dxMaskBit3 = 0x20
	dxMaskBit4 = 0x10
	dxMaskBit5 = 0x08
	dxMaskBit6 = 0x04
	dxMaskBit7 = 0x02
	dxMaskBit8 = 0x01
)

// Channel status constants.
const (
	dxChannelValid            = 0x00
	dxChannelUnsupportedChar  = 0x01
	dxChannelCannotUnlatch    = 0x02
)

// Unlatch type constants.
const (
	dxUnlatchExplicit = 0
	dxUnlatchImplicit = 1
)

// Module state flags.
const (
	dxModuleOff      = 0x00
	dxModuleOnRed    = 0x01
	dxModuleOnGreen  = 0x02
	dxModuleOnBlue   = 0x04
	dxModuleOnRGB    = 0x07
	dxModuleAssigned = 0x10
	dxModuleVisited  = 0x20
	dxModuleData     = 0x40
)

// Encoding character constants.
const (
	dxCharAsciiPad        = 129
	dxCharAsciiUpperShift = 235
	dxCharTripletShift1   = 0
	dxCharTripletShift2   = 1
	dxCharTripletShift3   = 2
	dxCharFNC1            = 232
	dxCharTripletUnlatch  = 254
	dxCharEdifactUnlatch  = 31
	dxCharC40Latch        = 230
	dxCharTextLatch       = 239
	dxCharX12Latch        = 238
	dxCharEdifactLatch    = 240
	dxCharBase256Latch    = 231
)

const (
	dxSzSquareCount = 24
	dxSzRectCount   = 6
	dxUndefined     = -1
)

// Symbol attribute tables (30 entries: 24 square + 6 rectangular).
var dxSymbolRows = [30]int{
	10, 12, 14, 16, 18, 20, 22, 24, 26,
	32, 36, 40, 44, 48, 52,
	64, 72, 80, 88, 96, 104,
	120, 132, 144,
	8, 8, 12, 12, 16, 16,
}

var dxSymbolCols = [30]int{
	10, 12, 14, 16, 18, 20, 22, 24, 26,
	32, 36, 40, 44, 48, 52,
	64, 72, 80, 88, 96, 104,
	120, 132, 144,
	18, 32, 26, 36, 36, 48,
}

var dxDataRegionRows = [30]int{
	8, 10, 12, 14, 16, 18, 20, 22, 24,
	14, 16, 18, 20, 22, 24,
	14, 16, 18, 20, 22, 24,
	18, 20, 22,
	6, 6, 10, 10, 14, 14,
}

var dxDataRegionCols = [30]int{
	8, 10, 12, 14, 16, 18, 20, 22, 24,
	14, 16, 18, 20, 22, 24,
	14, 16, 18, 20, 22, 24,
	18, 20, 22,
	16, 14, 24, 16, 16, 22,
}

var dxHorizDataRegions = [30]int{
	1, 1, 1, 1, 1, 1, 1, 1, 1,
	2, 2, 2, 2, 2, 2,
	4, 4, 4, 4, 4, 4,
	6, 6, 6,
	1, 2, 1, 2, 2, 2,
}

var dxInterleavedBlocks = [30]int{
	1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 2,
	2, 4, 4, 4, 4, 6,
	6, 8, 10,
	1, 1, 1, 1, 1, 1,
}

var dxSymbolDataWords = [30]int{
	3, 5, 8, 12, 18, 22, 30, 36, 44,
	62, 86, 114, 144, 174, 204,
	280, 368, 456, 576, 696, 816,
	1050, 1304, 1558,
	5, 10, 16, 22, 32, 49,
}

var dxBlockErrorWords = [30]int{
	5, 7, 10, 12, 14, 18, 20, 24, 28,
	36, 42, 48, 56, 68, 42,
	56, 36, 48, 56, 68, 56,
	68, 62, 62,
	7, 11, 14, 18, 24, 28,
}

var dxBlockMaxCorrectable = [30]int{
	2, 3, 5, 6, 7, 9, 10, 12, 14,
	18, 21, 24, 28, 34, 21,
	28, 18, 24, 28, 34, 28,
	34, 31, 31,
	3, 5, 7, 9, 12, 14,
}

// GF(256) antilog table.
var dxALogVal = [256]int{
	1, 2, 4, 8, 16, 32, 64, 128, 45, 90, 180, 69, 138, 57, 114, 228,
	229, 231, 227, 235, 251, 219, 155, 27, 54, 108, 216, 157, 23, 46, 92, 184,
	93, 186, 89, 178, 73, 146, 9, 18, 36, 72, 144, 13, 26, 52, 104, 208,
	141, 55, 110, 220, 149, 7, 14, 28, 56, 112, 224, 237, 247, 195, 171, 123,
	246, 193, 175, 115, 230, 225, 239, 243, 203, 187, 91, 182, 65, 130, 41, 82,
	164, 101, 202, 185, 95, 190, 81, 162, 105, 210, 137, 63, 126, 252, 213, 135,
	35, 70, 140, 53, 106, 212, 133, 39, 78, 156, 21, 42, 84, 168, 125, 250,
	217, 159, 19, 38, 76, 152, 29, 58, 116, 232, 253, 215, 131, 43, 86, 172,
	117, 234, 249, 223, 147, 11, 22, 44, 88, 176, 77, 154, 25, 50, 100, 200,
	189, 87, 174, 113, 226, 233, 255, 211, 139, 59, 118, 236, 245, 199, 163, 107,
	214, 129, 47, 94, 188, 85, 170, 121, 242, 201, 191, 83, 166, 97, 194, 169,
	127, 254, 209, 143, 51, 102, 204, 181, 71, 142, 49, 98, 196, 165, 103, 206,
	177, 79, 158, 17, 34, 68, 136, 61, 122, 244, 197, 167, 99, 198, 161, 111,
	222, 145, 15, 30, 60, 120, 240, 205, 183, 67, 134, 33, 66, 132, 37, 74,
	148, 5, 10, 20, 40, 80, 160, 109, 218, 153, 31, 62, 124, 248, 221, 151,
	3, 6, 12, 24, 48, 96, 192, 173, 119, 238, 241, 207, 179, 75, 150, 1,
}

// GF(256) log table.
var dxLogVal = [256]int{
	-255, 255, 1, 240, 2, 225, 241, 53, 3, 38, 226, 133, 242, 43, 54, 210,
	4, 195, 39, 114, 227, 106, 134, 28, 243, 140, 44, 23, 55, 118, 211, 234,
	5, 219, 196, 96, 40, 222, 115, 103, 228, 78, 107, 125, 135, 8, 29, 162,
	244, 186, 141, 180, 45, 99, 24, 49, 56, 13, 119, 153, 212, 199, 235, 91,
	6, 76, 220, 217, 197, 11, 97, 184, 41, 36, 223, 253, 116, 138, 104, 193,
	229, 86, 79, 171, 108, 165, 126, 145, 136, 34, 9, 74, 30, 32, 163, 84,
	245, 173, 187, 204, 142, 81, 181, 190, 46, 88, 100, 159, 25, 231, 50, 207,
	57, 147, 14, 67, 120, 128, 154, 248, 213, 167, 200, 63, 236, 110, 92, 176,
	7, 161, 77, 124, 221, 102, 218, 95, 198, 90, 12, 152, 98, 48, 185, 179,
	42, 209, 37, 132, 224, 52, 254, 239, 117, 233, 139, 22, 105, 27, 194, 113,
	230, 206, 87, 158, 80, 189, 172, 203, 109, 175, 166, 62, 127, 247, 146, 66,
	137, 192, 35, 252, 10, 183, 75, 216, 31, 83, 33, 73, 164, 144, 85, 170,
	246, 65, 174, 61, 188, 202, 205, 157, 143, 169, 82, 72, 182, 215, 191, 251,
	47, 178, 89, 151, 101, 94, 160, 123, 26, 112, 232, 21, 51, 238, 208, 131,
	58, 69, 148, 18, 15, 16, 68, 17, 121, 149, 129, 19, 155, 59, 249, 70,
	214, 250, 168, 71, 201, 156, 64, 60, 237, 130, 111, 20, 93, 122, 177, 150,
}

// --- GF(256) helper functions ---

func dxGFProduct(a, b int) int {
	if a == 0 || b == 0 {
		return 0
	}
	return dxALogVal[(dxLogVal[a]+dxLogVal[b])%255]
}

func dxGFSum(a, b int) int {
	return a ^ b
}

func dxGFDoublify(a, b int) int {
	if a == 0 {
		return 0
	}
	if b == 0 {
		return a
	}
	return dxALogVal[(dxLogVal[a]+b)%255]
}

// --- Symbol attribute lookup ---

func dxGetSymbolAttribute(attribute, sizeIdx int) int {
	if sizeIdx < 0 || sizeIdx >= dxSzSquareCount+dxSzRectCount {
		return dxUndefined
	}
	switch attribute {
	case dxAttrSymbolRows:
		return dxSymbolRows[sizeIdx]
	case dxAttrSymbolCols:
		return dxSymbolCols[sizeIdx]
	case dxAttrDataRegionRows:
		return dxDataRegionRows[sizeIdx]
	case dxAttrDataRegionCols:
		return dxDataRegionCols[sizeIdx]
	case dxAttrHorizDataRegions:
		return dxHorizDataRegions[sizeIdx]
	case dxAttrVertDataRegions:
		if sizeIdx < dxSzSquareCount {
			return dxHorizDataRegions[sizeIdx]
		}
		return 1
	case dxAttrMappingMatrixRows:
		return dxDataRegionRows[sizeIdx] * dxGetSymbolAttribute(dxAttrVertDataRegions, sizeIdx)
	case dxAttrMappingMatrixCols:
		return dxDataRegionCols[sizeIdx] * dxHorizDataRegions[sizeIdx]
	case dxAttrInterleavedBlocks:
		return dxInterleavedBlocks[sizeIdx]
	case dxAttrBlockErrorWords:
		return dxBlockErrorWords[sizeIdx]
	case dxAttrBlockMaxCorrectable:
		return dxBlockMaxCorrectable[sizeIdx]
	case dxAttrSymbolDataWords:
		return dxSymbolDataWords[sizeIdx]
	case dxAttrSymbolErrorWords:
		return dxBlockErrorWords[sizeIdx] * dxInterleavedBlocks[sizeIdx]
	case dxAttrSymbolMaxCorrectable:
		return dxBlockMaxCorrectable[sizeIdx] * dxInterleavedBlocks[sizeIdx]
	}
	return dxUndefined
}

func dxGetBlockDataSize(sizeIdx, blockIdx int) int {
	sdw := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
	ilb := dxGetSymbolAttribute(dxAttrInterleavedBlocks, sizeIdx)
	if sdw < 1 || ilb < 1 {
		return dxUndefined
	}
	count := sdw / ilb
	if sizeIdx == DxSz144x144 && blockIdx < 8 {
		return count + 1
	}
	return count
}

// --- Reed-Solomon ECC generation ---

func dxGenReedSolECC(code []int, sizeIdx int) {
	sdw := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
	sew := dxGetSymbolAttribute(dxAttrSymbolErrorWords, sizeIdx)
	stw := sdw + sew
	bew := dxGetSymbolAttribute(dxAttrBlockErrorWords, sizeIdx)
	step := dxGetSymbolAttribute(dxAttrInterleavedBlocks, sizeIdx)

	if bew != sew/step {
		return
	}

	g := make([]int, 69)
	g[0] = 0x01
	for i := 1; i < 69; i++ {
		g[i] = 0x01
	}

	for i := 1; i <= bew; i++ {
		for j := i - 1; j >= 0; j-- {
			g[j] = dxGFDoublify(g[j], i)
			if j > 0 {
				g[j] = dxGFSum(g[j], g[j-1])
			}
		}
	}

	for block := 0; block < step; block++ {
		b := make([]int, 68)

		for i := block; i < sdw; i += step {
			val := dxGFSum(b[bew-1], code[i])
			for j := bew - 1; j > 0; j-- {
				b[j] = dxGFSum(b[j-1], dxGFProduct(g[j], val))
			}
			b[0] = dxGFProduct(g[0], val)
		}

		blockDataWords := dxGetBlockDataSize(sizeIdx, block)
		bIndex := bew
		for i := block + (step * blockDataWords); i < stw; i += step {
			bIndex--
			code[i] = b[bIndex]
		}
	}
}

// --- Randomize helpers ---

func dxRandomize253(codewordValue, codewordPosition int) int {
	pseudoRandom := ((149 * codewordPosition) % 253) + 1
	tmp := codewordValue + pseudoRandom
	if tmp > 254 {
		tmp -= 254
	}
	return tmp & 0xFF
}

func dxRandomize255(codewordValue, codewordPosition int) int {
	pseudoRandom := ((149 * codewordPosition) % 255) + 1
	tmp := codewordValue + pseudoRandom
	if tmp <= 255 {
		return tmp & 0xFF
	}
	return (tmp - 256) & 0xFF
}

func dxUnRandomize255(value, idx int) int {
	pseudoRandom := ((149 * idx) % 255) + 1
	tmp := value - pseudoRandom
	if tmp < 0 {
		tmp += 256
	}
	return tmp & 0xFF
}

// --- Triplet/Quadruplet helpers ---

func dxGetTripletValues(cw1, cw2 int) [3]int {
	compact := (cw1 << 8) | cw2
	return [3]int{
		(compact - 1) / 1600,
		((compact - 1) / 40) % 40,
		(compact - 1) % 40,
	}
}

func dxGetQuadrupletValues(cw1, cw2, cw3 int) [4]int {
	return [4]int{
		cw1 >> 2,
		((cw1 & 0x03) << 4) | ((cw2 & 0xf0) >> 4),
		((cw2 & 0x0f) << 2) | ((cw3 & 0xc0) >> 6),
		cw3 & 0x3f,
	}
}

// --- dxMessage ---

type dxMessage struct {
	array    []int
	code     []int
	padCount int
}

func newDxMessage(sizeIdx int) *dxMessage {
	mr := dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
	mc := dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)
	cs := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) +
		dxGetSymbolAttribute(dxAttrSymbolErrorWords, sizeIdx)
	return &dxMessage{
		array: make([]int, mc*mr),
		code:  make([]int, cs),
	}
}

func (m *dxMessage) symbolModuleStatus(sizeIdx, symbolRow, symbolCol int) int {
	drr := dxGetSymbolAttribute(dxAttrDataRegionRows, sizeIdx)
	drc := dxGetSymbolAttribute(dxAttrDataRegionCols, sizeIdx)
	sr := dxGetSymbolAttribute(dxAttrSymbolRows, sizeIdx)
	mc := dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

	srr := sr - symbolRow - 1
	mappingRow := srr - 1 - 2*(srr/(drr+2))
	mappingCol := symbolCol - 1 - 2*(symbolCol/(drc+2))

	// Solid finder pattern (L-shape: bottom and left edges)
	if symbolRow%(drr+2) == 0 || symbolCol%(drc+2) == 0 {
		return dxModuleOnRGB
	}

	// Alternating clock pattern (top edge)
	if (symbolRow+1)%(drr+2) == 0 {
		if (symbolCol & 0x01) != 0 {
			return 0
		}
		return dxModuleOnRGB
	}

	// Alternating clock pattern (right edge)
	if (symbolCol+1)%(drc+2) == 0 {
		if (symbolRow & 0x01) != 0 {
			return 0
		}
		return dxModuleOnRGB
	}

	// Data module
	return m.array[mappingRow*mc+mappingCol] | dxModuleData
}

// --- dxChannel ---

type dxChannel struct {
	encScheme     int
	invalid       int
	inputIndex    int
	encodedLength int
	currentLength int
	firstCodeWord int
	input         []int
	encodedWords  []int
}

func newDxChannel() *dxChannel {
	return &dxChannel{
		encScheme:    DxSchemeAscii,
		invalid:      dxChannelValid,
		encodedWords: make([]int, 1558),
	}
}

func copyDxChannel(ch *dxChannel) *dxChannel {
	nc := &dxChannel{
		encScheme:     ch.encScheme,
		invalid:       ch.invalid,
		inputIndex:    ch.inputIndex,
		encodedLength: ch.encodedLength,
		currentLength: ch.currentLength,
		firstCodeWord: ch.firstCodeWord,
		input:         ch.input, // shared read-only
		encodedWords:  make([]int, len(ch.encodedWords)),
	}
	copy(nc.encodedWords, ch.encodedWords)
	return nc
}

// --- dxEncode (main encoding engine) ---

type dxEncode struct {
	scheme        int
	sizeIdxReq    int
	marginSize    int
	moduleSize    int
	flgGS1        bool
	message       *dxMessage
	rawData       [][]bool
	sizeIdx       int
	symbolRows    int
	symbolCols    int
	mappingRows   int
	mappingCols   int
}

func newDxEncode() *dxEncode {
	return &dxEncode{
		scheme:     DxSchemeAscii,
		sizeIdxReq: DxSzAuto,
		marginSize: 10,
		moduleSize: 5,
	}
}

func (e *dxEncode) encodeDataMatrixRaw(inputBytes []int) error {
	buf := make([]int, 4096)
	sizeIdx := e.sizeIdxReq

	dataWordCount, sizeIdx, err := e.encodeDataCodewords(buf, inputBytes, sizeIdx)
	if err != nil {
		return err
	}
	if dataWordCount <= 0 {
		return fmt.Errorf("failed to encode data: data may be empty or contain invalid characters")
	}
	if sizeIdx == DxSzAuto || sizeIdx == DxSzRectAuto {
		return fmt.Errorf("data too long for specified symbol size")
	}

	paddedSize := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
	padCount := e.addPadChars(buf, dataWordCount, paddedSize)

	e.sizeIdx = sizeIdx
	e.symbolRows = dxGetSymbolAttribute(dxAttrSymbolRows, sizeIdx)
	e.symbolCols = dxGetSymbolAttribute(dxAttrSymbolCols, sizeIdx)
	e.mappingRows = dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
	e.mappingCols = dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

	e.message = newDxMessage(sizeIdx)
	e.message.padCount = padCount

	for i := 0; i < paddedSize; i++ {
		e.message.code[i] = buf[i]
	}

	dxGenReedSolECC(e.message.code, sizeIdx)
	e.modulePlacementECC200(e.message.array, e.message.code, sizeIdx, dxModuleOnRGB)
	e.printPatternRaw()

	return nil
}

// --- Encoding schemes ---

func (e *dxEncode) encodeDataCodewords(buf, inputBytes []int, sizeIdx int) (int, int, error) {
	var dwc int

	switch e.scheme {
	case DxSchemeAutoBest:
		dwc = e.encodeAutoBest(buf, inputBytes)
	case DxSchemeAutoFast:
		dwc = 0
	default:
		dwc = e.encodeSingleScheme(buf, inputBytes, e.scheme)
	}

	sizeIdx = e.findCorrectSymbolSize(dwc, sizeIdx)
	if sizeIdx == DxSzShapeAuto {
		return 0, sizeIdx, nil
	}
	return dwc, sizeIdx, nil
}

func (e *dxEncode) findCorrectSymbolSize(dataWords, sizeIdxRequest int) int {
	if dataWords <= 0 {
		return DxSzShapeAuto
	}

	sizeIdx := sizeIdxRequest

	if sizeIdxRequest == DxSzAuto || sizeIdxRequest == DxSzRectAuto {
		var idxBeg, idxEnd int
		if sizeIdxRequest == DxSzAuto {
			idxBeg = 0
			idxEnd = dxSzSquareCount
		} else {
			idxBeg = dxSzSquareCount
			idxEnd = dxSzSquareCount + dxSzRectCount
		}

		sizeIdx = idxBeg
		for i := idxBeg; i < idxEnd; i++ {
			sizeIdx = i
			if dxGetSymbolAttribute(dxAttrSymbolDataWords, i) >= dataWords {
				break
			}
		}

		if sizeIdx == idxEnd {
			return DxSzShapeAuto
		}
	}

	if dataWords > dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) {
		return DxSzShapeAuto
	}

	return sizeIdx
}

func (e *dxEncode) encodeSingleScheme(buf, codewords []int, scheme int) int {
	channel := newDxChannel()
	e.initChannel(channel, codewords)

	for channel.inputIndex < len(channel.input) {
		ok := e.encodeNextWord(channel, scheme)
		if !ok {
			return 0
		}
		if channel.invalid != dxChannelValid {
			return 0
		}
	}

	size := channel.encodedLength / 12
	for i := 0; i < size; i++ {
		buf[i] = channel.encodedWords[i]
	}
	return size
}

func (e *dxEncode) encodeAutoBest(buf, codewords []int) int {
	channels := make([]*dxChannel, 6)

	for ts := DxSchemeAscii; ts <= DxSchemeBase256; ts++ {
		channels[ts] = newDxChannel()
		e.initChannel(channels[ts], codewords)
		ok := e.encodeNextWord(channels[ts], ts)
		// C++ bug: "if (err) return 0;" - returns 0 when encoding succeeds
		if ok {
			return 0
		}
	}

	// Encode remaining (unreachable due to above bug, ported for faithfulness)
	for channels[0].inputIndex < len(channels[0].input) {
		best := make([]*dxChannel, 6)
		for ts := DxSchemeAscii; ts <= DxSchemeBase256; ts++ {
			best[ts] = e.findBestChannel(channels, ts)
		}
		channels = best
	}

	winner := channels[DxSchemeAscii]
	for ts := DxSchemeAscii + 1; ts <= DxSchemeBase256; ts++ {
		if channels[ts] == nil {
			continue
		}
		if channels[ts].invalid != dxChannelValid {
			continue
		}
		if channels[ts].encodedLength < winner.encodedLength {
			winner = channels[ts]
		}
	}

	ws := winner.encodedLength / 12
	for i := 0; i < ws; i++ {
		buf[i] = winner.encodedWords[i]
	}
	return ws
}

func (e *dxEncode) findBestChannel(channels []*dxChannel, targetScheme int) *dxChannel {
	var winner *dxChannel
	for s := DxSchemeAscii; s <= DxSchemeBase256; s++ {
		channel := channels[s]
		if channel == nil || channel.invalid != dxChannelValid {
			continue
		}
		if channel.inputIndex == len(channel.input) {
			continue
		}
		e.encodeNextWord(channel, targetScheme)
		if (channel.invalid & dxChannelUnsupportedChar) != 0 {
			return copyDxChannel(channel)
		}
		if (channel.invalid & dxChannelCannotUnlatch) != 0 {
			continue
		}
		if winner == nil || channel.currentLength < winner.currentLength {
			winner = copyDxChannel(channel)
		}
	}
	if winner != nil {
		return winner
	}
	return newDxChannel()
}

func (e *dxEncode) encodeNextWord(channel *dxChannel, targetScheme int) bool {
	if channel.encScheme != targetScheme {
		e.changeEncScheme(channel, targetScheme, dxUnlatchExplicit)
		if channel.invalid != dxChannelValid {
			return false
		}
	}

	if channel.encScheme != targetScheme {
		return false
	}

	switch channel.encScheme {
	case DxSchemeAscii:
		return e.encodeAsciiCodeword(channel)
	case DxSchemeC40, DxSchemeText, DxSchemeX12:
		return e.encodeTripletCodeword(channel)
	case DxSchemeEdifact:
		return e.encodeEdifactCodeword(channel)
	case DxSchemeBase256:
		return e.encodeBase256Codeword(channel)
	}
	return false
}

// --- ASCII encoding ---

func (e *dxEncode) encodeAsciiCodeword(channel *dxChannel) bool {
	inputValue := channel.input[channel.inputIndex]

	// Digit pair compression
	if isDigitByte(inputValue) && channel.currentLength >= channel.firstCodeWord+12 {
		prevIndex := (channel.currentLength - 12) / 12
		prevValue := (channel.encodedWords[prevIndex] - 1) & 0xFF
		prevPrevValue := 0
		if prevIndex > channel.firstCodeWord/12 {
			prevPrevValue = channel.encodedWords[prevIndex-1]
		}

		if prevPrevValue != 235 && isDigitByte(prevValue) {
			channel.encodedWords[prevIndex] = (10*(prevValue-48) + (inputValue - 48) + 130) & 0xFF
			channel.inputIndex++
			return true
		}
	}

	// FNC1 in GS1 mode
	if e.flgGS1 && inputValue == dxCharFNC1 {
		e.pushInputWord(channel, dxCharFNC1)
		e.incrementProgress(channel, 12)
		channel.inputIndex++
		return true
	}

	// Upper shift for values >= 128
	if inputValue >= 128 {
		e.pushInputWord(channel, dxCharAsciiUpperShift)
		e.incrementProgress(channel, 12)
		inputValue -= 128
	}

	e.pushInputWord(channel, (inputValue+1)&0xFF)
	e.incrementProgress(channel, 12)
	channel.inputIndex++
	return true
}

func isDigitByte(v int) bool {
	return v >= 48 && v <= 57
}

// --- Triplet encoding (C40/Text/X12) ---

func (e *dxEncode) encodeTripletCodeword(channel *dxChannel) bool {
	if channel.encScheme != DxSchemeC40 && channel.encScheme != DxSchemeText &&
		channel.encScheme != DxSchemeX12 {
		return false
	}

	if channel.currentLength > channel.encodedLength {
		return false
	}

	if channel.currentLength == channel.encodedLength {
		if channel.currentLength%12 != 0 {
			return false
		}

		ptrIndex := channel.inputIndex
		tripletCount := 0
		buffer := [6]int{}

		for {
			for tripletCount < 3 && ptrIndex < len(channel.input) {
				inputWord := channel.input[ptrIndex]
				ptrIndex++
				outputWords := e.getC40TextX12Words(inputWord, channel.encScheme)
				if len(outputWords) == 0 {
					channel.invalid = dxChannelUnsupportedChar
					return false
				}
				for _, w := range outputWords {
					buffer[tripletCount] = w & 0xFF
					tripletCount++
				}
			}

			triplet := [3]int{buffer[0], buffer[1], buffer[2]}

			if tripletCount >= 3 {
				e.pushTriplet(channel, triplet)
				buffer[0] = buffer[3]
				buffer[1] = buffer[4]
				buffer[2] = buffer[5]
				tripletCount -= 3
			}

			if ptrIndex == len(channel.input) {
				for channel.currentLength < channel.encodedLength {
					e.incrementProgress(channel, 8)
					channel.inputIndex++
				}

				if channel.currentLength == channel.encodedLength+8 {
					channel.currentLength = channel.encodedLength
					channel.inputIndex--
				}

				inputCount := len(channel.input) - channel.inputIndex
				ok := e.processEndOfSymbolTriplet(channel, triplet, tripletCount, inputCount)
				if !ok {
					return false
				}
				break
			}

			if tripletCount == 0 {
				break
			}
		}
	}

	if channel.currentLength < channel.encodedLength {
		e.incrementProgress(channel, 8)
		channel.inputIndex++
	}

	return true
}

func (e *dxEncode) getC40TextX12Words(inputWord, encScheme int) []int {
	var result []int

	if inputWord > 127 {
		if encScheme == DxSchemeX12 {
			return nil
		}
		result = append(result, dxCharTripletShift2)
		result = append(result, 30)
		inputWord -= 128
	}

	if encScheme == DxSchemeX12 {
		switch {
		case inputWord == 13:
			result = append(result, 0)
		case inputWord == 42:
			result = append(result, 1)
		case inputWord == 62:
			result = append(result, 2)
		case inputWord == 32:
			result = append(result, 3)
		case inputWord >= 45 && inputWord <= 57:
			result = append(result, inputWord-44)
		case inputWord >= 65 && inputWord <= 90:
			result = append(result, inputWord-51)
		}
	} else {
		switch {
		case inputWord <= 31:
			result = append(result, dxCharTripletShift1)
			result = append(result, inputWord)
		case inputWord == 32:
			result = append(result, 3)
		case inputWord <= 47:
			result = append(result, dxCharTripletShift2)
			result = append(result, inputWord-33)
		case inputWord <= 57:
			result = append(result, inputWord-44)
		case inputWord <= 64:
			result = append(result, dxCharTripletShift2)
			result = append(result, inputWord-43)
		case inputWord <= 90 && encScheme == DxSchemeC40:
			result = append(result, inputWord-51)
		case inputWord <= 90 && encScheme == DxSchemeText:
			result = append(result, dxCharTripletShift3)
			result = append(result, inputWord-64)
		case inputWord <= 95:
			result = append(result, dxCharTripletShift2)
			result = append(result, inputWord-69)
		case inputWord == 96 && encScheme == DxSchemeText:
			result = append(result, dxCharTripletShift3)
			result = append(result, 0)
		case inputWord <= 122 && encScheme == DxSchemeText:
			result = append(result, inputWord-83)
		case inputWord <= 127:
			result = append(result, dxCharTripletShift3)
			result = append(result, inputWord-96)
		}
	}

	return result
}

func (e *dxEncode) processEndOfSymbolTriplet(channel *dxChannel, triplet [3]int, tripletCount, inputCount int) bool {
	if channel.currentLength%12 != 0 {
		return false
	}

	inputAdjust := tripletCount - inputCount
	currentByte := channel.currentLength / 12
	needed := currentByte + inputCount
	if inputCount == 3 {
		needed = currentByte + 2
	}
	sizeIdx := e.findCorrectSymbolSize(needed, e.sizeIdxReq)
	if sizeIdx == DxSzShapeAuto {
		return false
	}

	remaining := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte

	if inputCount == 1 && remaining == 1 {
		e.changeEncScheme(channel, DxSchemeAscii, dxUnlatchImplicit)
		ok := e.encodeNextWord(channel, DxSchemeAscii)
		if !ok {
			return false
		}
		if channel.invalid != dxChannelValid || channel.inputIndex != len(channel.input) {
			return false
		}
	} else if remaining == 2 {
		if tripletCount == 3 {
			e.pushTriplet(channel, triplet)
			e.incrementProgress(channel, 24)
			channel.encScheme = DxSchemeAscii
			channel.inputIndex += 3
			channel.inputIndex -= inputAdjust
		} else if tripletCount == 2 {
			t := [3]int{triplet[0], triplet[1], 0}
			e.pushTriplet(channel, t)
			e.incrementProgress(channel, 24)
			channel.encScheme = DxSchemeAscii
			channel.inputIndex += 2
			channel.inputIndex -= inputAdjust
		} else if tripletCount == 1 {
			e.changeEncScheme(channel, DxSchemeAscii, dxUnlatchExplicit)
			ok := e.encodeNextWord(channel, DxSchemeAscii)
			if !ok {
				return false
			}
			if channel.invalid != dxChannelValid {
				return false
			}
		}
	} else {
		currentByte = channel.currentLength / 12
		remaining = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte
		if remaining > 0 {
			e.changeEncScheme(channel, DxSchemeAscii, dxUnlatchExplicit)
			for channel.inputIndex < len(channel.input) {
				ok := e.encodeNextWord(channel, DxSchemeAscii)
				if !ok {
					return false
				}
				if channel.invalid != dxChannelValid {
					return false
				}
			}
		}
	}

	return channel.inputIndex == len(channel.input)
}

func (e *dxEncode) pushTriplet(channel *dxChannel, triplet [3]int) {
	tv := 1600*triplet[0] + 40*triplet[1] + triplet[2] + 1
	e.pushInputWord(channel, (tv>>8)&0xFF)
	e.pushInputWord(channel, tv&0xFF)
}

// --- Edifact encoding ---

func (e *dxEncode) encodeEdifactCodeword(channel *dxChannel) bool {
	if channel.encScheme != DxSchemeEdifact {
		return false
	}

	inputValue := channel.input[channel.inputIndex]
	if inputValue < 32 || inputValue > 94 {
		channel.invalid = dxChannelUnsupportedChar
		return false
	}

	e.pushInputWord(channel, inputValue&0x3f)
	e.incrementProgress(channel, 9)
	channel.inputIndex++

	e.checkEndOfSymbolEdifact(channel)
	return true
}

func (e *dxEncode) checkEndOfSymbolEdifact(channel *dxChannel) {
	edifactValues := len(channel.input) - channel.inputIndex
	if edifactValues > 4 {
		return
	}

	currentByte := channel.currentLength / 12
	sizeIdx := e.findCorrectSymbolSize(currentByte, DxSzAuto)
	symbolCodewords := dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte

	if channel.currentLength%12 == 0 && (symbolCodewords == 1 || symbolCodewords == 2) {
		asciiCodewords := edifactValues
		if asciiCodewords <= symbolCodewords {
			e.changeEncScheme(channel, DxSchemeAscii, dxUnlatchImplicit)
			for i := 0; i < edifactValues; i++ {
				ok := e.encodeNextWord(channel, DxSchemeAscii)
				if !ok {
					return
				}
				if channel.invalid != dxChannelValid {
					return
				}
			}
		}
	} else if edifactValues == 0 {
		e.changeEncScheme(channel, DxSchemeAscii, dxUnlatchExplicit)
	}
}

// --- Base256 encoding ---

func (e *dxEncode) encodeBase256Codeword(channel *dxChannel) bool {
	if channel.encScheme != DxSchemeBase256 {
		return false
	}

	firstBytePtrIndex := channel.firstCodeWord / 12
	headerByte0 := dxUnRandomize255(
		channel.encodedWords[firstBytePtrIndex],
		channel.firstCodeWord/12+1)

	var newDataLength int
	if headerByte0 <= 249 {
		newDataLength = headerByte0
	} else {
		newDataLength = 250 * (headerByte0 - 249)
		newDataLength += dxUnRandomize255(
			channel.encodedWords[firstBytePtrIndex+1],
			channel.firstCodeWord/12+2)
	}

	newDataLength++

	var headerByteCount int
	var hb [2]int
	if newDataLength <= 249 {
		headerByteCount = 1
		hb[0] = newDataLength & 0xFF
		hb[1] = 0
	} else {
		headerByteCount = 2
		hb[0] = (newDataLength/250 + 249) & 0xFF
		hb[1] = (newDataLength % 250) & 0xFF
	}

	if newDataLength <= 0 || newDataLength > 1555 {
		return false
	}

	if newDataLength == 250 {
		for i := channel.currentLength/12 - 1; i > channel.firstCodeWord/12; i-- {
			valTmp := dxUnRandomize255(channel.encodedWords[i], i+1)
			channel.encodedWords[i+1] = dxRandomize255(valTmp, i+2)
		}
		e.incrementProgress(channel, 12)
		channel.encodedLength += 12
	}

	for i := 0; i < headerByteCount; i++ {
		channel.encodedWords[firstBytePtrIndex+i] = dxRandomize255(
			hb[i], channel.firstCodeWord/12+i+1)
	}

	e.pushInputWord(channel, dxRandomize255(
		channel.input[channel.inputIndex], channel.currentLength/12+1))
	e.incrementProgress(channel, 12)
	channel.inputIndex++
	return true
}

// --- Channel utilities ---

func (e *dxEncode) pushInputWord(channel *dxChannel, codeword int) {
	if channel.encodedLength/12 > 3*1558 {
		return
	}

	switch channel.encScheme {
	case DxSchemeAscii, DxSchemeC40, DxSchemeText, DxSchemeX12, DxSchemeBase256:
		channel.encodedWords[channel.currentLength/12] = codeword & 0xFF
		channel.encodedLength += 12

	case DxSchemeEdifact:
		pos := channel.currentLength % 4
		startByte := ((channel.currentLength + 9) / 12) - pos

		q := dxGetQuadrupletValues(
			channel.encodedWords[startByte],
			channel.encodedWords[startByte+1],
			channel.encodedWords[startByte+2])
		q[pos] = codeword & 0xFF
		for i := pos + 1; i < 4; i++ {
			q[i] = 0
		}

		if pos >= 2 {
			channel.encodedWords[startByte+2] = ((q[2] & 0x03) << 6) | q[3]
		}
		if pos >= 1 {
			channel.encodedWords[startByte+1] = ((q[1] & 0x0f) << 4) | (q[2] >> 2)
		}
		channel.encodedWords[startByte] = (q[0] << 2) | (q[1] >> 4)

		channel.encodedLength += 9
	}
}

func (e *dxEncode) incrementProgress(channel *dxChannel, encodedUnits int) {
	if channel.encScheme == DxSchemeC40 || channel.encScheme == DxSchemeText {
		pos := (channel.currentLength % 6) / 2
		startByte := (channel.currentLength / 12) - (pos >> 1)
		triplet := dxGetTripletValues(
			channel.encodedWords[startByte],
			channel.encodedWords[startByte+1])
		if triplet[pos] <= 2 {
			channel.currentLength += 8
		}
	}

	channel.currentLength += encodedUnits
}

func (e *dxEncode) changeEncScheme(channel *dxChannel, targetScheme, unlatchType int) {
	if channel.encScheme == targetScheme {
		return
	}

	switch channel.encScheme {
	case DxSchemeAscii:
		if channel.currentLength%12 != 0 && channel.currentLength%20 != 0 {
			return
		}

	case DxSchemeC40, DxSchemeText, DxSchemeX12:
		if channel.currentLength%12 != 0 && channel.currentLength%20 != 0 {
			channel.invalid = dxChannelCannotUnlatch
			return
		}
		if channel.currentLength != channel.encodedLength {
			channel.invalid = dxChannelCannotUnlatch
			return
		}
		if unlatchType == dxUnlatchExplicit {
			e.pushInputWord(channel, dxCharTripletUnlatch)
			e.incrementProgress(channel, 12)
		}

	case DxSchemeEdifact:
		if channel.currentLength%3 != 0 {
			return
		}
		if unlatchType == dxUnlatchExplicit {
			e.pushInputWord(channel, dxCharEdifactUnlatch)
			e.incrementProgress(channel, 9)
		}
		advance := (channel.currentLength % 4) * 3
		channel.currentLength += advance
		channel.encodedLength += advance

	// Base256: nothing on unlatch
	}

	channel.encScheme = DxSchemeAscii

	switch targetScheme {
	case DxSchemeAscii:
		// no latch needed
	case DxSchemeC40:
		e.pushInputWord(channel, dxCharC40Latch)
		e.incrementProgress(channel, 12)
	case DxSchemeText:
		e.pushInputWord(channel, dxCharTextLatch)
		e.incrementProgress(channel, 12)
	case DxSchemeX12:
		e.pushInputWord(channel, dxCharX12Latch)
		e.incrementProgress(channel, 12)
	case DxSchemeEdifact:
		e.pushInputWord(channel, dxCharEdifactLatch)
		e.incrementProgress(channel, 12)
	case DxSchemeBase256:
		e.pushInputWord(channel, dxCharBase256Latch)
		e.incrementProgress(channel, 12)
		e.pushInputWord(channel, dxRandomize255(0, 2))
		e.incrementProgress(channel, 12)
	}

	channel.encScheme = targetScheme
	channel.firstCodeWord = channel.currentLength - 12
}

func (e *dxEncode) initChannel(channel *dxChannel, codewords []int) {
	channel.encScheme = DxSchemeAscii
	channel.invalid = dxChannelValid
	channel.inputIndex = 0
	channel.input = make([]int, len(codewords))
	copy(channel.input, codewords)
}

// --- Padding ---

func (e *dxEncode) addPadChars(buf []int, dataWordCount, paddedSize int) int {
	padCount := 0
	if dataWordCount < paddedSize {
		padCount++
		buf[dataWordCount] = dxCharAsciiPad
		dataWordCount++
	}

	for dataWordCount < paddedSize {
		padCount++
		buf[dataWordCount] = dxRandomize253(dxCharAsciiPad, dataWordCount+1)
		dataWordCount++
	}

	return padCount
}

// --- Module placement ECC200 ---

func (e *dxEncode) modulePlacementECC200(modules, codewords []int, sizeIdx, moduleOnColor int) {
	mr := dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
	mc := dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

	ch := 0
	row := 4
	col := 0

	for {
		if row == mr && col == 0 {
			e.patternShapeSpecial1(modules, mr, mc, codewords, ch, moduleOnColor)
			ch++
		} else if row == mr-2 && col == 0 && mc%4 != 0 {
			e.patternShapeSpecial2(modules, mr, mc, codewords, ch, moduleOnColor)
			ch++
		} else if row == mr-2 && col == 0 && mc%8 == 4 {
			e.patternShapeSpecial3(modules, mr, mc, codewords, ch, moduleOnColor)
			ch++
		} else if row == mr+4 && col == 2 && mc%8 == 0 {
			e.patternShapeSpecial4(modules, mr, mc, codewords, ch, moduleOnColor)
			ch++
		}

		// Upward diagonal
		for {
			if row < mr && col >= 0 && (modules[row*mc+col]&dxModuleVisited) == 0 {
				e.patternShapeStandard(modules, mr, mc, row, col, codewords, ch, moduleOnColor)
				ch++
			}
			row -= 2
			col += 2
			if !(row >= 0 && col < mc) {
				break
			}
		}
		row += 1
		col += 3

		// Downward diagonal
		for {
			if row >= 0 && col < mc && (modules[row*mc+col]&dxModuleVisited) == 0 {
				e.patternShapeStandard(modules, mr, mc, row, col, codewords, ch, moduleOnColor)
				ch++
			}
			row += 2
			col -= 2
			if !(row < mr && col >= 0) {
				break
			}
		}
		row += 3
		col += 1

		if !(row < mr || col < mc) {
			break
		}
	}

	// Fill unvisited corner
	if (modules[mr*mc-1] & dxModuleVisited) == 0 {
		modules[mr*mc-1] |= moduleOnColor
		modules[mr*mc-mc-2] |= moduleOnColor
	}
}

func (e *dxEncode) patternShapeStandard(modules []int, mr, mc, row, col int, cw []int, ci, color int) {
	e.placeModule(modules, mr, mc, row-2, col-2, cw, ci, dxMaskBit1, color)
	e.placeModule(modules, mr, mc, row-2, col-1, cw, ci, dxMaskBit2, color)
	e.placeModule(modules, mr, mc, row-1, col-2, cw, ci, dxMaskBit3, color)
	e.placeModule(modules, mr, mc, row-1, col-1, cw, ci, dxMaskBit4, color)
	e.placeModule(modules, mr, mc, row-1, col, cw, ci, dxMaskBit5, color)
	e.placeModule(modules, mr, mc, row, col-2, cw, ci, dxMaskBit6, color)
	e.placeModule(modules, mr, mc, row, col-1, cw, ci, dxMaskBit7, color)
	e.placeModule(modules, mr, mc, row, col, cw, ci, dxMaskBit8, color)
}

func (e *dxEncode) patternShapeSpecial1(modules []int, mr, mc int, cw []int, ci, color int) {
	e.placeModule(modules, mr, mc, mr-1, 0, cw, ci, dxMaskBit1, color)
	e.placeModule(modules, mr, mc, mr-1, 1, cw, ci, dxMaskBit2, color)
	e.placeModule(modules, mr, mc, mr-1, 2, cw, ci, dxMaskBit3, color)
	e.placeModule(modules, mr, mc, 0, mc-2, cw, ci, dxMaskBit4, color)
	e.placeModule(modules, mr, mc, 0, mc-1, cw, ci, dxMaskBit5, color)
	e.placeModule(modules, mr, mc, 1, mc-1, cw, ci, dxMaskBit6, color)
	e.placeModule(modules, mr, mc, 2, mc-1, cw, ci, dxMaskBit7, color)
	e.placeModule(modules, mr, mc, 3, mc-1, cw, ci, dxMaskBit8, color)
}

func (e *dxEncode) patternShapeSpecial2(modules []int, mr, mc int, cw []int, ci, color int) {
	e.placeModule(modules, mr, mc, mr-3, 0, cw, ci, dxMaskBit1, color)
	e.placeModule(modules, mr, mc, mr-2, 0, cw, ci, dxMaskBit2, color)
	e.placeModule(modules, mr, mc, mr-1, 0, cw, ci, dxMaskBit3, color)
	e.placeModule(modules, mr, mc, 0, mc-4, cw, ci, dxMaskBit4, color)
	e.placeModule(modules, mr, mc, 0, mc-3, cw, ci, dxMaskBit5, color)
	e.placeModule(modules, mr, mc, 0, mc-2, cw, ci, dxMaskBit6, color)
	e.placeModule(modules, mr, mc, 0, mc-1, cw, ci, dxMaskBit7, color)
	e.placeModule(modules, mr, mc, 1, mc-1, cw, ci, dxMaskBit8, color)
}

func (e *dxEncode) patternShapeSpecial3(modules []int, mr, mc int, cw []int, ci, color int) {
	e.placeModule(modules, mr, mc, mr-3, 0, cw, ci, dxMaskBit1, color)
	e.placeModule(modules, mr, mc, mr-2, 0, cw, ci, dxMaskBit2, color)
	e.placeModule(modules, mr, mc, mr-1, 0, cw, ci, dxMaskBit3, color)
	e.placeModule(modules, mr, mc, 0, mc-2, cw, ci, dxMaskBit4, color)
	e.placeModule(modules, mr, mc, 0, mc-1, cw, ci, dxMaskBit5, color)
	e.placeModule(modules, mr, mc, 1, mc-1, cw, ci, dxMaskBit6, color)
	e.placeModule(modules, mr, mc, 2, mc-1, cw, ci, dxMaskBit7, color)
	e.placeModule(modules, mr, mc, 3, mc-1, cw, ci, dxMaskBit8, color)
}

func (e *dxEncode) patternShapeSpecial4(modules []int, mr, mc int, cw []int, ci, color int) {
	e.placeModule(modules, mr, mc, mr-1, 0, cw, ci, dxMaskBit1, color)
	e.placeModule(modules, mr, mc, mr-1, mc-1, cw, ci, dxMaskBit2, color)
	e.placeModule(modules, mr, mc, 0, mc-3, cw, ci, dxMaskBit3, color)
	e.placeModule(modules, mr, mc, 0, mc-2, cw, ci, dxMaskBit4, color)
	e.placeModule(modules, mr, mc, 0, mc-1, cw, ci, dxMaskBit5, color)
	e.placeModule(modules, mr, mc, 1, mc-3, cw, ci, dxMaskBit6, color)
	e.placeModule(modules, mr, mc, 1, mc-2, cw, ci, dxMaskBit7, color)
	e.placeModule(modules, mr, mc, 1, mc-1, cw, ci, dxMaskBit8, color)
}

func (e *dxEncode) placeModule(modules []int, mr, mc, row, col int, codeword []int, ci, mask, color int) {
	if row < 0 {
		row += mr
		col += 4 - ((mr + 4) % 8)
	}
	if col < 0 {
		col += mc
		row += 4 - ((mc + 4) % 8)
	}

	index := row*mc + col
	if (modules[index] & dxModuleAssigned) != 0 {
		if (modules[index] & color) != 0 {
			modules[index] = modules[index] | mask
		} else {
			modules[index] = modules[index] & (^mask & 0xFF)
		}
	} else {
		if (codeword[ci] & mask) != 0 {
			modules[index] |= color
		}
		modules[index] |= dxModuleAssigned
	}

	modules[index] |= dxModuleVisited
}

// --- Pattern output ---

func (e *dxEncode) printPatternRaw() {
	sc := e.symbolCols
	sr := e.symbolRows
	si := e.sizeIdx

	e.rawData = make([][]bool, sc)
	for c := 0; c < sc; c++ {
		e.rawData[c] = make([]bool, sr)
	}
	for symbolRow := 0; symbolRow < sr; symbolRow++ {
		for symbolCol := 0; symbolCol < sc; symbolCol++ {
			status := e.message.symbolModuleStatus(si, symbolRow, symbolCol)
			e.rawData[symbolCol][sr-symbolRow-1] = (status & dxModuleOnBlue) != 0
		}
	}
}

// ============================================================
// Public DataMatrix struct
// ============================================================

// DataMatrix generates DataMatrix ECC200 barcodes.
type DataMatrix struct {
	BarcodeBase2D
	codeSize     int
	encodeScheme int
}

// NewDataMatrix creates a new DataMatrix encoder with the given output format.
func NewDataMatrix(outputFormat string) *DataMatrix {
	dm := &DataMatrix{
		codeSize:     DxSzAuto,
		encodeScheme: DxSchemeAscii,
	}
	dm.InitBase2D(outputFormat)
	return dm
}

// SetCodeSize sets the DataMatrix symbol size (use DxSz* constants).
func (dm *DataMatrix) SetCodeSize(size int) {
	dm.codeSize = size
}

// GetCodeSize returns the current code size setting.
func (dm *DataMatrix) GetCodeSize() int {
	return dm.codeSize
}

// SetEncodeScheme sets the encoding scheme (use DxScheme* constants).
func (dm *DataMatrix) SetEncodeScheme(scheme int) {
	dm.encodeScheme = scheme
}

// GetEncodeScheme returns the current encoding scheme.
func (dm *DataMatrix) GetEncodeScheme() int {
	return dm.encodeScheme
}

// Draw renders the DataMatrix barcode at the given size.
// For DataMatrix, width and height are both controlled by size (square output).
func (dm *DataMatrix) Draw(code string, size int) error {
	data := stringToBytes(code)
	return dm.drawBytes(data, size, size)
}

// GetPattern returns the DataMatrix pattern as [][]bool (indexed [col][row]).
func (dm *DataMatrix) GetPattern(code string) ([][]bool, error) {
	data := stringToBytes(code)
	return dm.calDataMatrixBytes(data)
}

// drawBytes encodes byte data and renders to SVG or PNG.
func (dm *DataMatrix) drawBytes(data []int, width, height int) error {
	patt, err := dm.calDataMatrixBytes(data)
	if err != nil {
		return err
	}

	patternCols := len(patt)
	if patternCols == 0 {
		return fmt.Errorf("empty pattern")
	}
	patternRows := len(patt[0])
	if patternRows == 0 {
		return fmt.Errorf("empty pattern")
	}

	if dm.IsSVGOutput() {
		return dm.drawSVGDM(patt, width, height)
	}
	return dm.drawPNGDM(patt, width, height)
}

// calDataMatrixBytes encodes byte data and returns the raw pattern.
func (dm *DataMatrix) calDataMatrixBytes(byteData []int) ([][]bool, error) {
	enc := newDxEncode()
	enc.moduleSize = 1
	enc.marginSize = 0
	enc.sizeIdxReq = dm.codeSize
	enc.scheme = dm.encodeScheme

	err := enc.encodeDataMatrixRaw(byteData)
	if err != nil {
		return nil, err
	}
	if enc.rawData == nil {
		return nil, fmt.Errorf("encoding produced no pattern data")
	}
	return enc.rawData, nil
}

// stringToBytes converts a string to a byte slice as int values.
func stringToBytes(s string) []int {
	b := []byte(s)
	result := make([]int, len(b))
	for i, v := range b {
		result[i] = int(v)
	}
	return result
}

// --- SVG rendering ---

func (dm *DataMatrix) drawSVGDM(patt [][]bool, width, height int) error {
	pc := len(patt)
	pr := len(patt[0])

	var ms float64
	if dm.fitWidth {
		ms = math.Min(float64(width)/float64(pc), float64(height)/float64(pr))
	} else {
		ms = float64(minInt(width/pc, height/pr))
		if ms < 1.0 {
			ms = 1.0
		}
	}

	aw := int(math.Ceil(ms * float64(pc)))
	ah := int(math.Ceil(ms * float64(pr)))

	dm.svgBegin(aw, ah)
	dm.svgRect(0, 0, float64(aw), float64(ah), dm.backColor)

	adj := float64(dm.pxAdjBlack)
	for r := 0; r < pr; r++ {
		for c := 0; c < pc; c++ {
			if patt[c][r] {
				dw := ms + adj + 0.5
				dh := ms + adj + 0.5
				if dw < 0 {
					dw = 0
				}
				if dh < 0 {
					dh = 0
				}
				dm.svgRect(float64(c)*ms, float64(r)*ms, dw, dh, dm.foreColor)
			}
		}
	}

	dm.svgEnd()
	return nil
}

// --- PNG rendering ---

func (dm *DataMatrix) drawPNGDM(patt [][]bool, width, height int) error {
	pc := len(patt)
	pr := len(patt[0])

	var ms float64
	if dm.fitWidth {
		ms = math.Min(float64(width)/float64(pc), float64(height)/float64(pr))
	} else {
		ms = float64(minInt(width/pc, height/pr))
		if ms < 1.0 {
			ms = 1.0
		}
	}

	aw := int(math.Ceil(ms * float64(pc)))
	ah := int(math.Ceil(ms * float64(pr)))

	img := image.NewNRGBA(image.Rect(0, 0, aw, ah))
	draw.Draw(img, img.Bounds(), &image.Uniform{dm.backColor}, image.Point{}, draw.Src)

	for r := 0; r < pr; r++ {
		for c := 0; c < pc; c++ {
			if patt[c][r] {
				x0 := int(float64(c) * ms)
				y0 := int(float64(r) * ms)
				x1 := int(float64(c+1) * ms)
				y1 := int(float64(r+1) * ms)
				fillRect(img, x0, y0, x1, y1, dm.foreColor)
			}
		}
	}

	// Trial mode watermark
	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, aw, ah)
	}

	return dm.encodeImage(img)
}

func minInt(a, b int) int {
	if a < b {
		return a
	}
	return b
}
