package barcode

import (
	"fmt"
	"image"
	"image/draw"
	"math"
)

// drawMultiRowSVG renders a multi-row pattern-based barcode as SVG.
// patterns: run-length encoded strings for each row (digit = width, leading "0" means starts with space)
// rowHeights: height in X units for each row (-1 = variable/fills remaining space)
func drawMultiRowSVG(b *BarcodeBase, patterns []string, rowHeights []int, width, height int) error {
	if len(patterns) == 0 {
		return fmt.Errorf("no patterns to render")
	}

	// Calculate total fixed height and number of variable rows
	totalFixed := 0
	variableCount := 0
	for _, h := range rowHeights {
		if h > 0 {
			totalFixed += h
		} else {
			variableCount++
		}
	}

	// Calculate row pixel heights
	rowPxHeights := make([]int, len(rowHeights))
	if variableCount > 0 {
		variableHeight := (height - totalFixed) / variableCount
		if variableHeight < 1 {
			variableHeight = 1
		}
		for i, h := range rowHeights {
			if h > 0 {
				rowPxHeights[i] = h
			} else {
				rowPxHeights[i] = variableHeight
			}
		}
	} else {
		// Scale fixed heights proportionally
		scale := float64(height) / float64(totalFixed)
		for i, h := range rowHeights {
			rowPxHeights[i] = int(math.Round(float64(h) * scale))
		}
	}

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	y := 0
	for ri, pat := range patterns {
		if ri >= len(rowPxHeights) {
			break
		}
		rh := rowPxHeights[ri]
		drawPatternRowSVG(b, pat, 0, y, width, rh)
		y += rh
	}

	if IsTrialMode() {
		b.svgSampleOverlay(0, 0, width, height)
	}
	b.svgStream += "</svg>\n"
	return nil
}

// drawPatternRowSVG draws a single row from a run-length pattern string.
func drawPatternRowSVG(b *BarcodeBase, pattern string, x, y, width, height int) {
	if len(pattern) == 0 {
		return
	}

	// Parse pattern: leading "0" means starts with space
	startIdx := 0
	isBar := true
	if pattern[0] == '0' {
		isBar = false
		startIdx = 1
	}

	// Calculate total modules
	totalModules := 0
	for i := startIdx; i < len(pattern); i++ {
		totalModules += int(pattern[i] - '0')
	}
	if totalModules <= 0 {
		return
	}

	unitW := float64(width) / float64(totalModules)
	accum := 0.0
	bar := isBar

	for i := startIdx; i < len(pattern); i++ {
		modules := int(pattern[i] - '0')
		x1 := math.Round(float64(x) + accum*unitW)
		accum += float64(modules)
		x2 := math.Round(float64(x) + accum*unitW)
		if bar && x2 > x1 {
			b.svgRect(x1, float64(y), x2-x1, float64(height), b.foreColor)
		}
		bar = !bar
	}
}

// drawMultiRowPNG renders a multi-row pattern-based barcode as PNG.
func drawMultiRowPNG(b *BarcodeBase, patterns []string, rowHeights []int, width, height int) error {
	if len(patterns) == 0 {
		return fmt.Errorf("no patterns to render")
	}

	// Calculate total fixed height and number of variable rows
	totalFixed := 0
	variableCount := 0
	for _, h := range rowHeights {
		if h > 0 {
			totalFixed += h
		} else {
			variableCount++
		}
	}

	// Calculate row pixel heights
	rowPxHeights := make([]int, len(rowHeights))
	if variableCount > 0 {
		variableHeight := (height - totalFixed) / variableCount
		if variableHeight < 1 {
			variableHeight = 1
		}
		for i, h := range rowHeights {
			if h > 0 {
				rowPxHeights[i] = h
			} else {
				rowPxHeights[i] = variableHeight
			}
		}
	} else {
		scale := float64(height) / float64(totalFixed)
		for i, h := range rowHeights {
			rowPxHeights[i] = int(math.Round(float64(h) * scale))
		}
	}

	img := image.NewNRGBA(image.Rect(0, 0, width, height))
	draw.Draw(img, img.Bounds(), &image.Uniform{b.backColor}, image.Point{}, draw.Src)

	y := 0
	for ri, pat := range patterns {
		if ri >= len(rowPxHeights) {
			break
		}
		rh := rowPxHeights[ri]
		drawPatternRowPNG(img, pat, 0, y, width, rh, b.foreColor)
		y += rh
	}

	if IsTrialMode() {
		drawSampleOverlayPNG(img, 0, 0, width, height)
	}

	return b.encodeImage(img)
}

// drawPatternRowPNG draws a single row from a run-length pattern string.
func drawPatternRowPNG(img *image.NRGBA, pattern string, x, y, width, height int, foreColor RGBA) {
	if len(pattern) == 0 {
		return
	}

	startIdx := 0
	isBar := true
	if pattern[0] == '0' {
		isBar = false
		startIdx = 1
	}

	totalModules := 0
	for i := startIdx; i < len(pattern); i++ {
		totalModules += int(pattern[i] - '0')
	}
	if totalModules <= 0 {
		return
	}

	unitW := float64(width) / float64(totalModules)
	accum := 0.0
	bar := isBar

	for i := startIdx; i < len(pattern); i++ {
		modules := int(pattern[i] - '0')
		x1 := int(math.Round(float64(x) + accum*unitW))
		accum += float64(modules)
		x2 := int(math.Round(float64(x) + accum*unitW))
		if bar && x2 > x1 {
			fillRect(img, x1, y, x2, y+height, foreColor)
		}
		bar = !bar
	}
}

// padLeft left-pads a string with '0' to the given length.
func padLeft(s string, length int) string {
	for len(s) < length {
		s = "0" + s
	}
	return s
}

// parseInt64 parses a numeric string to int64.
func parseInt64(s string) int64 {
	var v int64
	for _, c := range s {
		v = v*10 + int64(c-'0')
	}
	return v
}
