package barcode

import (
	"fmt"
	"strings"
)

var (
	code93Chars    = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%<>?!"
	code93Patterns = []string{
		"131112", "111213", "111312", "111411", "121113", "121212", "121311",
		"111114", "131211", "141111", "211113", "211212", "211311", "221112",
		"221211", "231111", "112113", "112212", "112311", "122112", "132111",
		"111123", "111222", "111321", "121122", "131121", "212112", "212211",
		"211122", "211221", "221121", "222111", "112122", "112221", "122121",
		"123111", "121131", "311112", "311211", "321111", "112131", "113121",
		"211131", "121221", "312111", "311121", "122211",
	}
	code93Start = "111141"
	code93Stop  = "1111411"
)

// Code93 encoder.
type Code93 struct {
	BarcodeBase1D
}

// NewCode93 creates a new Code93 encoder.
func NewCode93(outputFormat string) *Code93 {
	c := &Code93{}
	c.InitBase1D(outputFormat)
	return c
}

func code93Modulus47(code string, weight int) (int, error) {
	total := 0
	for i := 0; i < len(code); i++ {
		value := strings.IndexByte(code93Chars, code[i])
		if value == -1 {
			return 0, fmt.Errorf("invalid character in CODE93: %c", code[i])
		}
		posWeight := len(code) - i
		if posWeight > weight {
			posWeight = ((posWeight - 1) % weight) + 1
		}
		total += value * posWeight
	}
	return total % 47, nil
}

// Encode returns the bar/space width pattern for Code93.
func (c *Code93) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	code = strings.ToUpper(code)

	for _, ch := range code {
		if strings.IndexRune(code93Chars, ch) == -1 {
			return nil, fmt.Errorf("invalid character in CODE93: %c", ch)
		}
	}

	checkC, err := code93Modulus47(code, 20)
	if err != nil {
		return nil, err
	}
	codeWithC := code + string(code93Chars[checkC])

	checkK, err := code93Modulus47(codeWithC, 15)
	if err != nil {
		return nil, err
	}

	var result []int

	// Start pattern
	for _, ch := range code93Start {
		result = append(result, int(ch-'0'))
	}

	// Data
	for _, ch := range code {
		index := strings.IndexRune(code93Chars, ch)
		for _, p := range code93Patterns[index] {
			result = append(result, int(p-'0'))
		}
	}

	// Check digit C
	for _, p := range code93Patterns[checkC] {
		result = append(result, int(p-'0'))
	}

	// Check digit K
	for _, p := range code93Patterns[checkK] {
		result = append(result, int(p-'0'))
	}

	// Stop pattern
	for _, ch := range code93Stop {
		result = append(result, int(ch-'0'))
	}

	return result, nil
}

// Draw renders the Code93 barcode.
func (c *Code93) Draw(code string, width, height int) error {
	return c.BarcodeBase1D.Draw(code, width, height, c)
}
