package barcode

import (
	"strings"
	"testing"
)

func TestCode39Encode(t *testing.T) {
	c := NewCode39(FormatSVG)
	pattern, err := c.Encode("CODE39")
	if err != nil {
		t.Fatalf("Encode failed: %v", err)
	}
	if len(pattern) == 0 {
		t.Fatal("pattern is empty")
	}
	// Code39: * + 6 chars + * = 8 chars, each 9 bars + 1 gap between
	// total elements = 8*9 + 7*1 = 79
	if len(pattern) != 79 {
		t.Errorf("expected 79 elements, got %d", len(pattern))
	}
}

func TestCode39InvalidChar(t *testing.T) {
	c := NewCode39(FormatSVG)
	_, err := c.Encode("abc!")
	if err == nil {
		t.Fatal("expected error for invalid character")
	}
}

func TestCode39StarReserved(t *testing.T) {
	c := NewCode39(FormatSVG)
	_, err := c.Encode("A*B")
	if err == nil {
		t.Fatal("expected error for '*' character")
	}
}

func TestCode39Empty(t *testing.T) {
	c := NewCode39(FormatSVG)
	_, err := c.Encode("")
	if err == nil {
		t.Fatal("expected error for empty string")
	}
}

func TestCode39DrawSVG(t *testing.T) {
	c := NewCode39(FormatSVG)
	err := c.Draw("CODE39", 400, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	svg, err := c.GetSVG()
	if err != nil {
		t.Fatalf("GetSVG failed: %v", err)
	}
	if !strings.Contains(svg, "<svg") {
		t.Error("SVG output missing <svg tag")
	}
	if !strings.Contains(svg, "</svg>") {
		t.Error("SVG output missing </svg> tag")
	}
}

func TestCode39DrawPNG(t *testing.T) {
	c := NewCode39(FormatPNG)
	err := c.Draw("CODE39", 400, 100)
	if err != nil {
		t.Fatalf("Draw failed: %v", err)
	}
	data := c.GetImageMemory()
	if len(data) == 0 {
		t.Fatal("PNG output is empty")
	}
	// Check PNG magic bytes
	if data[0] != 0x89 || data[1] != 0x50 {
		t.Error("output is not valid PNG")
	}
}
