package barcode

import (
	"fmt"
	"strings"
)

// Code39 character set and 9-bit patterns.
var (
	code39Chars = "1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZ-. *$/+%"
	code39Ptn   = []int{
		0x121, 0x061, 0x160, 0x031, 0x130, 0x070, 0x025, 0x124, 0x064, 0x034,
		0x109, 0x049, 0x148, 0x019, 0x118, 0x058, 0x00d, 0x10c, 0x04c, 0x01c,
		0x103, 0x043, 0x142, 0x013, 0x112, 0x052, 0x007, 0x106, 0x046, 0x016,
		0x181, 0x0c1, 0x1c0, 0x091, 0x190, 0x0d0, 0x085, 0x184, 0x0c4, 0x094,
		0x0a8, 0x0a2, 0x08a, 0x02a,
	}
)

// Code39 encoder.
type Code39 struct {
	BarcodeBase1D
	ShowStartStop bool
}

// NewCode39 creates a new Code39 encoder.
func NewCode39(outputFormat string) *Code39 {
	c := &Code39{ShowStartStop: true}
	c.InitBase1D(outputFormat)
	return c
}

// Encode returns the bar/space width pattern for Code39.
func (c *Code39) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}

	var validCode strings.Builder
	for _, ch := range code {
		upper := strings.ToUpper(string(ch))
		if upper == "*" {
			return nil, fmt.Errorf("invalid character in CODE39: '*' is reserved for start/stop code")
		}
		pos := strings.Index(code39Chars, upper)
		if pos == -1 || pos == 39 { // 39 is '*'
			return nil, fmt.Errorf("invalid character in CODE39: %c", ch)
		}
		validCode.WriteString(upper)
	}

	fullCode := "*" + validCode.String() + "*"

	var result []int
	for _, ch := range fullCode {
		if len(result) > 0 {
			result = append(result, 1) // inter-character gap
		}
		charIndex := strings.IndexRune(code39Chars, ch)
		bitPattern := code39Ptn[charIndex]

		// 9-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
		mask := 0x100
		for i := 0; i < 9; i++ {
			if (bitPattern & (mask >> i)) != 0 {
				result = append(result, 3)
			} else {
				result = append(result, 1)
			}
		}
	}
	return result, nil
}

// Draw renders the Code39 barcode.
func (c *Code39) Draw(code string, width, height int) error {
	return c.BarcodeBase1D.Draw(code, width, height, c)
}
