package barcode

import "fmt"

// Code128 mode constants.
const (
	Code128Auto  = 0
	Code128CodeA = 1
	Code128CodeB = 2
	Code128CodeC = 3
)

var code128Patterns = []string{
	"212222", "222122", "222221", "121223", "121322", "131222", "122213",
	"122312", "132212", "221213", "221312", "231212", "112232", "122132",
	"122231", "113222", "123122", "123221", "223211", "221132", "221231",
	"213212", "223112", "312131", "311222", "321122", "321221", "312212",
	"322112", "322211", "212123", "212321", "232121", "111323", "131123",
	"131321", "112313", "132113", "132311", "211313", "231113", "231311",
	"112133", "112331", "132131", "113123", "113321", "133121", "313121",
	"211331", "231131", "213113", "213311", "213131", "311123", "311321",
	"331121", "312113", "312311", "332111", "314111", "221411", "431111",
	"111224", "111422", "121124", "121421", "141122", "141221", "112214",
	"112412", "122114", "122411", "142112", "142211", "241211", "221114",
	"413111", "241112", "134111", "111242", "121142", "121241", "114212",
	"124112", "124211", "411212", "421112", "421211", "212141", "214121",
	"412121", "111143", "111341", "131141", "114113", "114311", "411113",
	"411311", "113141", "114131", "311141", "411131",
	"211412", "211214", "211232", "2331112", // 103-106: Start A/B/C, Stop
}

const dpInf = 999999999

// Code128 encoder with AUTO/A/B/C mode support.
type Code128 struct {
	BarcodeBase1D
	CodeMode int // Code128Auto, Code128CodeA, Code128CodeB, Code128CodeC
}

// NewCode128 creates a new Code128 encoder.
func NewCode128(outputFormat string) *Code128 {
	c := &Code128{CodeMode: Code128Auto}
	c.InitBase1D(outputFormat)
	return c
}

func code128AIndex(c byte) int {
	if c <= 95 {
		return int(c)
	}
	return -1
}

func code128BIndex(c byte) int {
	if c < 32 || c > 126 {
		return -1
	}
	return int(c) - 32
}

func code128CIndex(s string, i int) int {
	if i+1 >= len(s) {
		return -1
	}
	c1, c2 := s[i], s[i+1]
	if c1 < '0' || c1 > '9' || c2 < '0' || c2 > '9' {
		return -1
	}
	return int(c1-'0')*10 + int(c2-'0')
}

const (
	dpSetA = 0
	dpSetB = 1
	dpSetC = 2
)

type dpResult struct {
	cost int
	enc  []int
}

func solveDP(s string, i int, iset int, memo map[int]*dpResult) (*dpResult, error) {
	if i >= len(s) {
		return &dpResult{0, nil}, nil
	}
	key := (i << 2) + iset
	if r, ok := memo[key]; ok {
		return r, nil
	}

	bestCost := dpInf
	var bestEnc []int
	c := s[i]

	// Stay in current set
	if iset == dpSetA {
		idx := code128AIndex(c)
		if idx >= 0 {
			nxt, err := solveDP(s, i+1, iset, memo)
			if err != nil {
				return nil, err
			}
			if cost := 1 + nxt.cost; cost < bestCost {
				bestCost = cost
				bestEnc = append([]int{idx}, nxt.enc...)
			}
		}
	} else if iset == dpSetB {
		idx := code128BIndex(c)
		if idx >= 0 {
			nxt, err := solveDP(s, i+1, iset, memo)
			if err != nil {
				return nil, err
			}
			if cost := 1 + nxt.cost; cost < bestCost {
				bestCost = cost
				bestEnc = append([]int{idx}, nxt.enc...)
			}
		}
	} else { // dpSetC
		idx := code128CIndex(s, i)
		if idx >= 0 {
			nxt, err := solveDP(s, i+2, iset, memo)
			if err != nil {
				return nil, err
			}
			if cost := 1 + nxt.cost; cost < bestCost {
				bestCost = cost
				bestEnc = append([]int{idx}, nxt.enc...)
			}
		}
	}

	// Switch to different set
	switchVals := [3][3]int{
		{0, 100, 99},   // from A: to A(unused), to B=100, to C=99
		{101, 0, 99},   // from B: to A=101, to B(unused), to C=99
		{101, 100, 0},  // from C: to A=101, to B=100, to C(unused)
	}

	for st := 0; st < 3; st++ {
		if st == iset {
			continue
		}
		sv := switchVals[iset][st]

		if st == dpSetA {
			idx := code128AIndex(c)
			if idx >= 0 {
				nxt, err := solveDP(s, i+1, st, memo)
				if err != nil {
					return nil, err
				}
				if cost := 2 + nxt.cost; cost < bestCost {
					bestCost = cost
					bestEnc = append([]int{sv, idx}, nxt.enc...)
				}
			}
		} else if st == dpSetB {
			idx := code128BIndex(c)
			if idx >= 0 {
				nxt, err := solveDP(s, i+1, st, memo)
				if err != nil {
					return nil, err
				}
				if cost := 2 + nxt.cost; cost < bestCost {
					bestCost = cost
					bestEnc = append([]int{sv, idx}, nxt.enc...)
				}
			}
		} else { // dpSetC
			idx := code128CIndex(s, i)
			if idx >= 0 {
				nxt, err := solveDP(s, i+2, st, memo)
				if err != nil {
					return nil, err
				}
				if cost := 2 + nxt.cost; cost < bestCost {
					bestCost = cost
					bestEnc = append([]int{sv, idx}, nxt.enc...)
				}
			}
		}
	}

	if bestCost == dpInf {
		return nil, fmt.Errorf("DP encode error at pos=%d", i)
	}

	r := &dpResult{bestCost, bestEnc}
	memo[key] = r
	return r, nil
}

// Encode returns the bar/space width pattern for Code128.
func (c *Code128) Encode(code string) ([]int, error) {
	if code == "" {
		return nil, fmt.Errorf("empty string")
	}
	if c.CodeMode == Code128Auto {
		return c.encodeAuto(code)
	}
	return c.encodeSingleSet(code, c.CodeMode)
}

func (c *Code128) encodeSingleSet(s string, fixedSet int) ([]int, error) {
	startVal := 102 + fixedSet // A=103, B=104, C=105
	vals := []int{startVal}

	i := 0
	for i < len(s) {
		ch := s[i]
		switch fixedSet {
		case Code128CodeC:
			cc := code128CIndex(s, i)
			if cc < 0 {
				return nil, fmt.Errorf("not 2-digit at pos=%d", i)
			}
			vals = append(vals, cc)
			i += 2
		case Code128CodeA:
			idx := code128AIndex(ch)
			if idx < 0 {
				return nil, fmt.Errorf("invalid char for CODE_A")
			}
			vals = append(vals, idx)
			i++
		default: // Code128CodeB
			idx := code128BIndex(ch)
			if idx < 0 {
				return nil, fmt.Errorf("invalid char for CODE_B")
			}
			vals = append(vals, idx)
			i++
		}
	}

	// Checksum (Modulus 103)
	checkSum := vals[0]
	for j := 1; j < len(vals); j++ {
		checkSum += vals[j] * j
	}
	checkSum %= 103
	vals = append(vals, checkSum)
	vals = append(vals, 106) // Stop

	return expandCode128Pattern(vals)
}

func (c *Code128) encodeAuto(s string) ([]int, error) {
	type candidate struct {
		startVal int
		cost     int
		enc      []int
	}

	var candidates []candidate
	for iset := 0; iset < 3; iset++ {
		startVal := 103 + iset
		memo := make(map[int]*dpResult)
		r, err := solveDP(s, 0, iset, memo)
		if err == nil {
			candidates = append(candidates, candidate{startVal, r.cost, r.enc})
		}
	}

	if len(candidates) == 0 {
		return nil, fmt.Errorf("no valid start code")
	}

	// Pick lowest cost
	best := candidates[0]
	for _, c := range candidates[1:] {
		if 1+c.cost < 1+best.cost {
			best = c
		}
	}

	encVals := append([]int{best.startVal}, best.enc...)

	// Checksum
	checkSum := encVals[0]
	for j := 1; j < len(encVals); j++ {
		checkSum += encVals[j] * j
	}
	ck := checkSum % 103
	encVals = append(encVals, ck)
	encVals = append(encVals, 106) // Stop

	return expandCode128Pattern(encVals)
}

func expandCode128Pattern(vals []int) ([]int, error) {
	var result []int
	for _, v := range vals {
		if v < 0 || v > 106 {
			return nil, fmt.Errorf("invalid code value %d", v)
		}
		ptn := code128Patterns[v]
		for _, ch := range ptn {
			result = append(result, int(ch-'0'))
		}
	}
	return result, nil
}

// Draw renders the Code128 barcode.
func (c *Code128) Draw(code string, width, height int) error {
	return c.BarcodeBase1D.Draw(code, width, height, c)
}
