package barcode

import (
	"fmt"
	"math"
)

// BarcodeBase1D is the base for 1D barcode symbologies.
type BarcodeBase1D struct {
	BarcodeBase
	ShowText                    bool
	TextEvenSpacing             bool
	textFontScale               float64
	textHorizontalSpacingScale  float64
	textVerticalOffsetScale     float64
	minLineWidth                int
}

// InitBase1D initialises BarcodeBase1D with defaults.
func (b *BarcodeBase1D) InitBase1D(outputFormat string) {
	b.InitBase(outputFormat)
	b.ShowText = true
	b.TextEvenSpacing = true
	b.textFontScale = 1.0
	b.textHorizontalSpacingScale = 1.0
	b.textVerticalOffsetScale = 1.0
	b.minLineWidth = 1
}

// SetTextFontScale sets the font scale for barcode text.
func (b *BarcodeBase1D) SetTextFontScale(s float64) { b.textFontScale = s }

// TextFontScale returns the font scale.
func (b *BarcodeBase1D) TextFontScale() float64 { return b.textFontScale }

// SetTextVerticalOffsetScale sets the vertical offset scale.
func (b *BarcodeBase1D) SetTextVerticalOffsetScale(s float64) { b.textVerticalOffsetScale = s }

// TextVerticalOffsetScale returns the vertical offset scale.
func (b *BarcodeBase1D) TextVerticalOffsetScale() float64 { return b.textVerticalOffsetScale }

// SetTextHorizontalSpacingScale sets the horizontal spacing scale.
func (b *BarcodeBase1D) SetTextHorizontalSpacingScale(s float64) {
	b.textHorizontalSpacingScale = s
}

// TextHorizontalSpacingScale returns the horizontal spacing scale.
func (b *BarcodeBase1D) TextHorizontalSpacingScale() float64 { return b.textHorizontalSpacingScale }

// SetMinLineWidth sets the minimum line width (for ITF, Matrix2of5, NEC2of5).
func (b *BarcodeBase1D) SetMinLineWidth(w int) { b.minLineWidth = w }

// MinLineWidth returns the minimum line width.
func (b *BarcodeBase1D) MinLineWidth() int { return b.minLineWidth }

// Encoder1D is implemented by each 1D barcode type.
type Encoder1D interface {
	Encode(code string) ([]int, error)
}

// Draw renders the 1D barcode to the internal buffer (SVG or PNG).
func (b *BarcodeBase1D) Draw(code string, width, height int, encoder Encoder1D) error {
	pattern, err := encoder.Encode(code)
	if err != nil {
		return err
	}
	if len(pattern) == 0 {
		return fmt.Errorf("empty pattern")
	}
	if b.IsSVGOutput() {
		return b.drawSVG1D(pattern, code, width, height)
	}
	return b.drawPNG1D(pattern, code, width, height)
}

// --- SVG rendering for generic 1D ---

func (b *BarcodeBase1D) drawSVG1D(pattern []int, code string, width, height int) error {
	display := ""
	if b.ShowText {
		display = code
	}
	return b.drawSVGBars(pattern, width, height, display)
}

func (b *BarcodeBase1D) drawSVGBars(pattern []int, width, height int, text string) error {
	totalUnits := 0
	for _, v := range pattern {
		totalUnits += v
	}
	if totalUnits <= 0 {
		return fmt.Errorf("pattern has zero total units")
	}

	fontSize := int(math.Max(8, float64(height)*0.15*b.textFontScale))

	textH := 0
	if text != "" {
		textH = int(float64(fontSize) * 1.4)
	}
	barH := height - textH
	if barH <= 0 {
		barH = height
		text = ""
	}

	b.svgBegin(width, height)
	b.svgRect(0, 0, float64(width), float64(height), b.backColor)

	unitW := float64(width) / float64(totalUnits)
	accum := 0.0
	isBar := true
	for _, units := range pattern {
		x1 := math.Round(accum * unitW)
		accum += float64(units)
		x2 := math.Round(accum * unitW)
		if isBar && x2 > x1 {
			b.svgRect(x1, 0, x2-x1, float64(barH), b.foreColor)
		}
		isBar = !isBar
	}

	if text != "" {
		textY := float64(barH) + 2
		if b.TextEvenSpacing {
			b.svgTextEvenDistribution(text, 0, textY, float64(width), fontSize, b.textHorizontalSpacingScale)
		} else {
			b.svgText(float64(width)/2, textY, text, fontSize, b.foreColor, "middle")
		}
	}

	b.svgEnd()
	return nil
}

// svgTextEvenDistribution renders text with even character distribution.
func (b *BarcodeBase1D) svgTextEvenDistribution(text string, x, y, width float64, fontSize int, hScale float64) {
	if text == "" {
		return
	}
	numChars := len(text)

	marginRatio := 0.05
	if width < 100 {
		marginRatio = 0.03
	} else if width > 400 {
		marginRatio = 0.07
	}

	horizontalMargin := width * marginRatio * hScale
	textWidth := width - 2*horizontalMargin

	charSpacing := textWidth
	if numChars > 1 {
		charSpacing = textWidth / float64(numChars-1)
	}

	minSpacing := float64(fontSize) / 2
	if charSpacing < minSpacing {
		charSpacing = minSpacing
	}

	totalTextWidth := 0.0
	if numChars > 1 {
		totalTextWidth = charSpacing * float64(numChars-1)
	}
	startX := x + (width-totalTextWidth)/2

	for i := 0; i < numChars; i++ {
		charX := startX + float64(i)*charSpacing
		b.svgText(charX, y, string(text[i]), fontSize, b.foreColor, "middle")
	}
}
