package barcode

import (
	"encoding/base64"
	"fmt"
	"image/color"
)

// RGBA represents a color with red, green, blue, and alpha components.
type RGBA = color.NRGBA

// BarcodeBase holds common state for all barcode types.
type BarcodeBase struct {
	foreColor   RGBA
	backColor   RGBA
	format      string
	fitWidth    bool
	pxAdjBlack  int
	pxAdjWhite  int
	svgStream   string
	svgWidth    int
	svgHeight   int
	imageBuffer []byte
}

// InitBase initialises BarcodeBase with defaults.
func (b *BarcodeBase) InitBase(outputFormat string) {
	b.foreColor = RGBA{0, 0, 0, 255}
	b.backColor = RGBA{255, 255, 255, 255}
	b.format = outputFormat
}

// SetForegroundColor sets the bar colour.
func (b *BarcodeBase) SetForegroundColor(r, g, b2, a uint8) {
	b.foreColor = RGBA{r, g, b2, a}
}

// SetBackgroundColor sets the background colour.
func (b *BarcodeBase) SetBackgroundColor(r, g, b2, a uint8) {
	b.backColor = RGBA{r, g, b2, a}
}

// ForeColor returns the foreground colour.
func (b *BarcodeBase) ForeColor() RGBA { return b.foreColor }

// BackColor returns the background colour.
func (b *BarcodeBase) BackColor() RGBA { return b.backColor }

// SetPxAdjustBlack sets pixel adjustment for black bars.
func (b *BarcodeBase) SetPxAdjustBlack(adj int) { b.pxAdjBlack = adj }

// SetPxAdjustWhite sets pixel adjustment for white spaces.
func (b *BarcodeBase) SetPxAdjustWhite(adj int) { b.pxAdjWhite = adj }

// SetFitWidth sets whether the barcode should fit the width exactly.
func (b *BarcodeBase) SetFitWidth(fit bool) { b.fitWidth = fit }

// FitWidth returns the current fit-width setting.
func (b *BarcodeBase) FitWidth() bool { return b.fitWidth }

// SetOutputFormat sets the output format ("png", "jpeg", "svg").
func (b *BarcodeBase) SetOutputFormat(fmt string) { b.format = fmt }

// OutputFormat returns the current output format.
func (b *BarcodeBase) OutputFormat() string { return b.format }

// IsSVGOutput returns true if the output format is SVG.
func (b *BarcodeBase) IsSVGOutput() bool { return b.format == FormatSVG }

// GetImageMemory returns the rendered image bytes (PNG/JPEG).
func (b *BarcodeBase) GetImageMemory() []byte {
	if b.imageBuffer != nil {
		return b.imageBuffer
	}
	return nil
}

// GetImageBase64 returns a data-URI string for PNG/JPEG output.
func (b *BarcodeBase) GetImageBase64() (string, error) {
	if b.IsSVGOutput() {
		return "", fmt.Errorf("GetImageBase64() is not available in SVG mode; use GetSVG()")
	}
	data := b.GetImageMemory()
	if len(data) == 0 {
		return "", nil
	}
	encoded := base64.StdEncoding.EncodeToString(data)
	mime := "image/png"
	if b.format == FormatJPEG {
		mime = "image/jpeg"
	}
	return fmt.Sprintf("data:%s;base64,%s", mime, encoded), nil
}

// GetSVG returns the SVG string.
func (b *BarcodeBase) GetSVG() (string, error) {
	if !b.IsSVGOutput() {
		return "", fmt.Errorf("GetSVG() requires SVG output mode")
	}
	return b.svgStream, nil
}

// colorToRGB returns an SVG rgb() string.
func colorToRGB(c RGBA) string {
	return fmt.Sprintf("rgb(%d,%d,%d)", c.R, c.G, c.B)
}

// --- SVG helpers ---

func (b *BarcodeBase) svgBegin(width, height int) {
	b.svgWidth = width
	b.svgHeight = height
	b.svgStream = ""
	b.svgStream += "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
	b.svgStream += fmt.Sprintf(
		"<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" "+
			"width=\"%d\" height=\"%d\" viewBox=\"0 0 %d %d\">\n",
		width, height, width, height)
}

func (b *BarcodeBase) svgEnd() {
	if IsTrialMode() {
		b.svgSampleOverlay(0, 0, b.svgWidth, b.svgHeight)
	}
	b.svgStream += "</svg>\n"
}

func (b *BarcodeBase) svgSampleOverlay(x, y, width, height int) {
	fontSize := int(float64(height) * 0.12)
	if fontSize < 8 {
		fontSize = 8
	}
	if fontSize > 40 {
		fontSize = 40
	}
	margin := int(float64(height) * 0.01)
	if margin < 2 {
		margin = 2
	}
	red := RGBA{255, 0, 0, 255}
	text := getTrialText()
	b.svgStream += fmt.Sprintf(
		"  <text x=\"%d\" y=\"%d\" font-family=\"Arial, sans-serif\" font-size=\"%d\" "+
			"font-weight=\"bold\" font-style=\"italic\" fill=\"%s\" "+
			"dominant-baseline=\"hanging\">%s</text>\n",
		x+margin, y+margin, fontSize, colorToRGB(red), text)
}

func (b *BarcodeBase) svgRect(x, y, w, h float64, c RGBA) {
	opacity := ""
	if c.A < 255 {
		opacity = fmt.Sprintf("\" fill-opacity=\"%.2f", float64(c.A)/255.0)
	}
	b.svgStream += fmt.Sprintf(
		"  <rect x=\"%.2f\" y=\"%.2f\" width=\"%.2f\" height=\"%.2f\" "+
			"fill=\"%s%s\"/>\n",
		x, y, w, h, colorToRGB(c), opacity)
}

func (b *BarcodeBase) svgText(x, y float64, text string, fontSize int, c RGBA, anchor string) {
	if anchor == "" {
		anchor = "middle"
	}
	b.svgStream += fmt.Sprintf(
		"  <text x=\"%.2f\" y=\"%.2f\" font-family=\"Arial, sans-serif\" "+
			"font-size=\"%d\" fill=\"%s\" text-anchor=\"%s\" "+
			"dominant-baseline=\"hanging\">%s</text>\n",
		x, y, fontSize, colorToRGB(c), anchor, text)
}
