package barcode

import (
	"fmt"
	"strings"
)

// BarcodeDataType identifies the rendering category.
type BarcodeDataType int

const (
	DataType1D       BarcodeDataType = iota // Generic 1D bar/space pattern
	DataTypeJANUPC                          // JAN/UPC with extended guard bars
	DataType4State                          // Postal 4-state bars (YubinCustomer)
	DataType2DMatrix                        // 2D matrix (QR, DataMatrix, PDF417)
	DataTypeMultiRow                        // Multi-row pattern (DataBar stacked)
)

// BarcodeData holds all data needed by external renderers.
type BarcodeData struct {
	Type BarcodeDataType

	// Common
	ForeColor RGBA
	BackColor RGBA

	// 1D properties
	Pattern           []int
	DisplayCode       string
	ShowText          bool
	TextEvenSpacing   bool
	TextFontScale     float64
	TextHSpacingScale float64
	TextVOffsetScale  float64

	// JAN/UPC specific
	JANUPCType    string // "JAN8", "JAN13", "UPCA", "UPCE"
	ExtendedGuard bool

	// 4-state
	Bars4State []int // bar types: 1-4

	// 2D matrix
	Matrix   [][]bool
	ColMajor bool // true=QR/DataMatrix (pattern[col][row]), false=PDF417 (pattern[row][col])

	// Multi-row DataBar
	MultiRowPatterns []string
	MultiRowHeights  []int
}

// EncodeData encodes a barcode and returns BarcodeData for external renderers.
// The bc parameter must be a pointer to one of the 18 barcode types.
func EncodeData(bc interface{}, code string) (*BarcodeData, error) {
	switch v := bc.(type) {

	// --- 2D Matrix types ---
	case *QRCode:
		patt, err := v.GetPattern(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:      DataType2DMatrix,
			ForeColor: v.foreColor,
			BackColor: v.backColor,
			Matrix:    patt,
			ColMajor:  true,
		}, nil

	case *DataMatrix:
		patt, err := v.GetPattern(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:      DataType2DMatrix,
			ForeColor: v.foreColor,
			BackColor: v.backColor,
			Matrix:    patt,
			ColMajor:  true,
		}, nil

	case *PDF417:
		patt, err := v.GetPattern(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:      DataType2DMatrix,
			ForeColor: v.foreColor,
			BackColor: v.backColor,
			Matrix:    patt,
			ColMajor:  false, // PDF417: pattern[row][col]
		}, nil

	// --- 4-State postal ---
	case *YubinCustomer:
		_, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:      DataType4State,
			ForeColor: v.foreColor,
			BackColor: v.backColor,
			Bars4State: v.Bars(),
		}, nil

	// --- JAN/UPC with extended guard bars ---
	case *JAN8:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		s := code
		if len(s) == 7 {
			s = s + CalculateCheckDigitJAN8(s)
		}
		return &BarcodeData{
			Type:              DataTypeJANUPC,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       s,
			ShowText:          v.ShowText,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
			JANUPCType:        "JAN8",
			ExtendedGuard:     v.ExtendedGuard,
		}, nil

	case *JAN13:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		s := code
		if len(s) == 12 {
			s = s + CalculateCheckDigitJAN13(s)
		}
		return &BarcodeData{
			Type:              DataTypeJANUPC,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       s,
			ShowText:          v.ShowText,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
			JANUPCType:        "JAN13",
			ExtendedGuard:     v.ExtendedGuard,
		}, nil

	case *UPCA:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		s := code
		if len(s) == 11 {
			s = s + CalculateCheckDigitUPCA(s)
		}
		return &BarcodeData{
			Type:              DataTypeJANUPC,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       s,
			ShowText:          v.ShowText,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
			JANUPCType:        "UPCA",
			ExtendedGuard:     v.ExtendedGuard,
		}, nil

	case *UPCE:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		displayCode := code
		if len(displayCode) == 6 {
			displayCode = "0" + displayCode
			displayCode += CalculateCheckDigitUPCE(displayCode)
		} else if len(displayCode) == 7 {
			displayCode += CalculateCheckDigitUPCE(displayCode)
		}
		return &BarcodeData{
			Type:              DataTypeJANUPC,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       displayCode,
			ShowText:          v.ShowText,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
			JANUPCType:        "UPCE",
			ExtendedGuard:     v.ExtendedGuard,
		}, nil

	// --- Multi-row DataBar variants ---
	case *GS1DataBar14:
		_, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		if v.symbolType == Omnidirectional {
			// Single-row: treat as multi-row with one row
			return &BarcodeData{
				Type:             DataTypeMultiRow,
				ForeColor:        v.foreColor,
				BackColor:        v.backColor,
				MultiRowPatterns: v.Patterns(),
				MultiRowHeights:  v.RowHeights(),
			}, nil
		}
		// Stacked / StackedOmnidirectional
		return &BarcodeData{
			Type:             DataTypeMultiRow,
			ForeColor:        v.foreColor,
			BackColor:        v.backColor,
			MultiRowPatterns: v.Patterns(),
			MultiRowHeights:  v.RowHeights(),
		}, nil

	case *GS1DataBarExpanded:
		_, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:             DataTypeMultiRow,
			ForeColor:        v.foreColor,
			BackColor:        v.backColor,
			MultiRowPatterns: v.Patterns(),
			MultiRowHeights:  v.RowHeights(),
		}, nil

	// --- GS1-128 (generic 1D with text cleanup) ---
	case *GS1128:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		display := ""
		if v.ShowText {
			display = strings.ReplaceAll(code, "{FNC1}", "")
			display = strings.ReplaceAll(display, "{AI}", "")
		}
		return &BarcodeData{
			Type:              DataType1D,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       display,
			ShowText:          v.ShowText,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
		}, nil

	// --- GS1 DataBar Limited (single-row 1D) ---
	case *GS1DataBarLimited:
		pattern, err := v.Encode(code)
		if err != nil {
			return nil, err
		}
		return &BarcodeData{
			Type:              DataType1D,
			ForeColor:         v.foreColor,
			BackColor:         v.backColor,
			Pattern:           pattern,
			DisplayCode:       "",
			ShowText:          false,
			TextEvenSpacing:   v.TextEvenSpacing,
			TextFontScale:     v.textFontScale,
			TextHSpacingScale: v.textHorizontalSpacingScale,
			TextVOffsetScale:  v.textVerticalOffsetScale,
		}, nil

	// --- Generic 1D types ---
	case *Code39:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *Code93:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *Code128:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *NW7:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *ITF:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *Matrix2of5:
		return encode1D(&v.BarcodeBase1D, v, code)
	case *NEC2of5:
		return encode1D(&v.BarcodeBase1D, v, code)

	default:
		return nil, fmt.Errorf("unsupported barcode type: %T", bc)
	}
}

// encode1D is a helper for generic 1D barcode types.
func encode1D(b *BarcodeBase1D, encoder Encoder1D, code string) (*BarcodeData, error) {
	pattern, err := encoder.Encode(code)
	if err != nil {
		return nil, err
	}
	display := ""
	if b.ShowText {
		display = code
	}
	return &BarcodeData{
		Type:              DataType1D,
		ForeColor:         b.foreColor,
		BackColor:         b.backColor,
		Pattern:           pattern,
		DisplayCode:       display,
		ShowText:          b.ShowText,
		TextEvenSpacing:   b.TextEvenSpacing,
		TextFontScale:     b.textFontScale,
		TextHSpacingScale: b.textHorizontalSpacingScale,
		TextVOffsetScale:  b.textVerticalOffsetScale,
	}, nil
}
