// Barcode.Go - All-in-One
//
// Full 18-type barcode REST API with 4 output modes using net/http.
//
//	go run main.go
//	→ http://localhost:5701
package main

import (
	"bytes"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"image/color"
	"image/png"
	"log"
	"math"
	"math/rand"
	"net/http"
	"os"
	"strconv"
	"text/template"
	"time"

	"github.com/fogleman/gg"
	barcode "github.com/pao-company/barcode-go"
	barcodeGg "github.com/pao-company/barcode-go/ggrenderer"
	barcodeGopdf "github.com/pao-company/barcode-go/gopdf"
	"github.com/signintech/gopdf"
)

// loadFont tries to load a TrueType font into the GoPdf instance.
func loadFont(pdf *gopdf.GoPdf) {
	fontPaths := []string{
		"C:/Windows/Fonts/arial.ttf",
		"/usr/share/fonts/dejavu-sans-fonts/DejaVuSans.ttf",
		"/usr/share/fonts/liberation-sans/LiberationSans-Regular.ttf",
		"/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf",
	}
	for _, fp := range fontPaths {
		if _, err := os.Stat(fp); err == nil {
			if err := pdf.AddTTFFont("font", fp); err == nil {
				pdf.SetFont("font", "", 12)
				return
			}
		}
	}
}

// barcodeTypeDef defines a barcode type for the UI.
type barcodeTypeDef struct {
	ID      string `json:"id"`
	Label   string `json:"label"`
	Group   string `json:"group"`
	Dim     string `json:"dim"`
	Default string `json:"default"`
	W       int    `json:"w"`
	H       int    `json:"h"`
}

var barcodeTypes = []barcodeTypeDef{
	// 2D
	{ID: "QR", Label: "QR Code", Group: "2D Barcode", Dim: "2d", Default: "https://www.pao.ac/", W: 200, H: 200},
	{ID: "DataMatrix", Label: "DataMatrix", Group: "2D Barcode", Dim: "2d", Default: "Hello DataMatrix", W: 200, H: 200},
	{ID: "PDF417", Label: "PDF417", Group: "2D Barcode", Dim: "2d", Default: "Hello PDF417", W: 200, H: 100},
	// Special
	{ID: "GS1_128", Label: "GS1-128", Group: "Special", Dim: "1d", Default: "[01]04912345123459", W: 400, H: 100},
	{ID: "YubinCustomer", Label: "Yubin Customer", Group: "Special", Dim: "postal", Default: "1060032", W: 400, H: 60},
	// 1D
	{ID: "Code128", Label: "Code 128", Group: "1D Barcode", Dim: "1d", Default: "Hello-2026", W: 400, H: 100},
	{ID: "Code39", Label: "Code 39", Group: "1D Barcode", Dim: "1d", Default: "HELLO-123", W: 400, H: 100},
	{ID: "Code93", Label: "Code 93", Group: "1D Barcode", Dim: "1d", Default: "CODE93", W: 400, H: 100},
	{ID: "NW7", Label: "NW-7 / Codabar", Group: "1D Barcode", Dim: "1d", Default: "A123456B", W: 400, H: 100},
	{ID: "ITF", Label: "ITF", Group: "1D Barcode", Dim: "1d", Default: "123456", W: 400, H: 100},
	{ID: "Matrix2of5", Label: "Matrix 2 of 5", Group: "1D Barcode", Dim: "1d", Default: "1234", W: 400, H: 100},
	{ID: "NEC2of5", Label: "NEC 2 of 5", Group: "1D Barcode", Dim: "1d", Default: "1234", W: 400, H: 100},
	// GS1 DataBar
	{ID: "GS1DataBar14", Label: "GS1 DataBar 14", Group: "GS1 DataBar", Dim: "1d", Default: "0123456789012", W: 300, H: 80},
	{ID: "GS1DataBarLimited", Label: "GS1 DataBar Limited", Group: "GS1 DataBar", Dim: "1d", Default: "0123456789012", W: 300, H: 80},
	{ID: "GS1DataBarExpanded", Label: "GS1 DataBar Expanded", Group: "GS1 DataBar", Dim: "1d", Default: "[01]90012345678908", W: 400, H: 80},
	// JAN / UPC
	{ID: "JAN13", Label: "JAN-13 / EAN-13", Group: "JAN / UPC", Dim: "1d", Default: "490123456789", W: 300, H: 100},
	{ID: "JAN8", Label: "JAN-8 / EAN-8", Group: "JAN / UPC", Dim: "1d", Default: "1234567", W: 250, H: 100},
	{ID: "UPCA", Label: "UPC-A", Group: "JAN / UPC", Dim: "1d", Default: "01234567890", W: 300, H: 100},
	{ID: "UPCE", Label: "UPC-E", Group: "JAN / UPC", Dim: "1d", Default: "0123456", W: 250, H: 100},
}

var barcodeMap map[string]barcodeTypeDef
var tmpl *template.Template

func init() {
	barcodeMap = make(map[string]barcodeTypeDef)
	for _, bt := range barcodeTypes {
		barcodeMap[bt.ID] = bt
	}
	tmpl = template.Must(template.ParseFiles("templates/index.html"))
}

func main() {
	http.HandleFunc("/", handleIndex)
	http.HandleFunc("/api/types", handleTypes)
	http.HandleFunc("/draw-base64", handleDrawBase64)
	http.HandleFunc("/draw-svg", handleDrawSVG)
	http.HandleFunc("/draw-canvas", handleDrawCanvas)
	http.HandleFunc("/pdf", handlePDF)

	fmt.Println("Barcode.Go All-in-One")
	fmt.Println("→ http://localhost:5701")
	log.Fatal(http.ListenAndServe(":5701", nil))
}

type pageData struct {
	BarcodeTypesJSON string
}

func handleIndex(w http.ResponseWriter, r *http.Request) {
	jsonBytes, _ := json.Marshal(barcodeTypes)
	tmpl.Execute(w, pageData{BarcodeTypesJSON: string(jsonBytes)})
}

func handleTypes(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(barcodeTypes)
}

type apiResponse struct {
	OK     bool   `json:"ok"`
	Base64 string `json:"base64,omitempty"`
	SVG    string `json:"svg,omitempty"`
	Error  string `json:"error,omitempty"`
}

func writeJSON(w http.ResponseWriter, resp apiResponse) {
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(resp)
}

func getFormInt(r *http.Request, key string, def int) int {
	s := r.FormValue(key)
	if s == "" {
		return def
	}
	v, err := strconv.Atoi(s)
	if err != nil {
		return def
	}
	return v
}

func getFormFloat(r *http.Request, key string, def float64) float64 {
	s := r.FormValue(key)
	if s == "" {
		return def
	}
	v, err := strconv.ParseFloat(s, 64)
	if err != nil {
		return def
	}
	return v
}

func parseHexColor(hex string) (uint8, uint8, uint8) {
	if len(hex) == 6 {
		r, _ := strconv.ParseUint(hex[0:2], 16, 8)
		g, _ := strconv.ParseUint(hex[2:4], 16, 8)
		b, _ := strconv.ParseUint(hex[4:6], 16, 8)
		return uint8(r), uint8(g), uint8(b)
	}
	return 0, 0, 0
}

// drawableBarcode is a common interface for barcode types.
type drawableBarcode interface {
	GetImageBase64() (string, error)
	GetSVG() (string, error)
}

func createAndDraw(typeID, code, outputFormat string, width, height int, r *http.Request) (drawableBarcode, error) {
	foreColor := r.FormValue("fore_color")
	backColor := r.FormValue("back_color")
	transparentBG := r.FormValue("transparent_bg") == "1"

	switch typeID {
	case "QR":
		bc := barcode.NewQRCode(outputFormat)
		eccLevel := getFormInt(r, "qr_error_level", 1)
		bc.SetErrorCorrectionLevel(eccLevel)
		ver := getFormInt(r, "qr_version", 0)
		if ver > 0 {
			bc.SetVersion(ver)
		}
		applyColors2D(&bc.BarcodeBase2D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width)

	case "DataMatrix":
		bc := barcode.NewDataMatrix(outputFormat)
		sz := getFormInt(r, "datamatrix_size", -1)
		if sz >= 0 {
			bc.SetCodeSize(sz)
		}
		applyColors2D(&bc.BarcodeBase2D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width)

	case "PDF417":
		bc := barcode.NewPDF417(outputFormat)
		el := getFormInt(r, "pdf417_error_level", -1)
		if el >= 0 && el <= 8 {
			bc.SetErrorLevel(el)
		}
		cols := getFormInt(r, "pdf417_cols", 0)
		if cols > 0 {
			bc.SetColumns(cols)
		}
		applyColors2D(&bc.BarcodeBase2D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "Code128":
		bc := barcode.NewCode128(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "Code39":
		bc := barcode.NewCode39(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "Code93":
		bc := barcode.NewCode93(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "NW7":
		bc := barcode.NewNW7(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "ITF":
		bc := barcode.NewITF(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "Matrix2of5":
		bc := barcode.NewMatrix2of5(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "NEC2of5":
		bc := barcode.NewNEC2of5(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "JAN13":
		bc := barcode.NewJAN13(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		if r.FormValue("extended_guard") == "1" {
			bc.ExtendedGuard = true
		} else if r.FormValue("extended_guard") == "0" {
			bc.ExtendedGuard = false
		}
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "JAN8":
		bc := barcode.NewJAN8(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		if r.FormValue("extended_guard") == "1" {
			bc.ExtendedGuard = true
		} else if r.FormValue("extended_guard") == "0" {
			bc.ExtendedGuard = false
		}
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "UPCA":
		bc := barcode.NewUPCA(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		if r.FormValue("extended_guard") == "1" {
			bc.ExtendedGuard = true
		} else if r.FormValue("extended_guard") == "0" {
			bc.ExtendedGuard = false
		}
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "UPCE":
		bc := barcode.NewUPCE(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		if r.FormValue("extended_guard") == "1" {
			bc.ExtendedGuard = true
		} else if r.FormValue("extended_guard") == "0" {
			bc.ExtendedGuard = false
		}
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "GS1_128":
		bc := barcode.NewGS1128(outputFormat)
		applyCommon1D(&bc.BarcodeBase1D, r)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "GS1DataBar14":
		symType := barcode.Omnidirectional
		st := r.FormValue("databar14_type")
		switch st {
		case "Stacked":
			symType = barcode.Stacked
		case "StackedOmni":
			symType = barcode.StackedOmnidirectional
		}
		bc := barcode.NewGS1DataBar14(outputFormat, symType)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "GS1DataBarLimited":
		bc := barcode.NewGS1DataBarLimited(outputFormat)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "GS1DataBarExpanded":
		symType := barcode.Unstacked
		if r.FormValue("databar_expanded_type") == "Stacked" {
			symType = barcode.StackedExp
		}
		numCols := getFormInt(r, "databar_expanded_cols", 2)
		bc := barcode.NewGS1DataBarExpanded(outputFormat, symType, numCols)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	case "YubinCustomer":
		bc := barcode.NewYubinCustomer(outputFormat)
		applyColors1D(&bc.BarcodeBase1D, foreColor, backColor, transparentBG)
		return bc, bc.Draw(code, width, height)

	default:
		return nil, fmt.Errorf("unknown barcode type: %s", typeID)
	}
}

func applyCommon1D(b *barcode.BarcodeBase1D, r *http.Request) {
	if r.FormValue("show_text") == "0" {
		b.ShowText = false
	}
	if r.FormValue("even_spacing") == "0" {
		b.TextEvenSpacing = false
	}
}

func applyColors2D(b *barcode.BarcodeBase2D, foreColor, backColor string, transparent bool) {
	if foreColor != "" && foreColor != "000000" {
		r, g, bl := parseHexColor(foreColor)
		b.SetForegroundColor(r, g, bl, 255)
	}
	if transparent {
		b.SetBackgroundColor(0, 0, 0, 0)
	} else if backColor != "" && backColor != "FFFFFF" {
		r, g, bl := parseHexColor(backColor)
		b.SetBackgroundColor(r, g, bl, 255)
	}
}

func applyColors1D(b *barcode.BarcodeBase1D, foreColor, backColor string, transparent bool) {
	if foreColor != "" && foreColor != "000000" {
		r, g, bl := parseHexColor(foreColor)
		b.SetForegroundColor(r, g, bl, 255)
	}
	if transparent {
		b.SetBackgroundColor(0, 0, 0, 0)
	} else if backColor != "" && backColor != "FFFFFF" {
		r, g, bl := parseHexColor(backColor)
		b.SetBackgroundColor(r, g, bl, 255)
	}
}

func handleDrawBase64(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}
	width := getFormInt(r, "width", info.W)
	height := getFormInt(r, "height", info.H)

	bc, err := createAndDraw(typeID, code, barcode.FormatPNG, width, height, r)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	b64, err := bc.GetImageBase64()
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: b64})
}

func handleDrawSVG(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}
	width := getFormInt(r, "width", info.W)
	height := getFormInt(r, "height", info.H)

	bc, err := createAndDraw(typeID, code, barcode.FormatSVG, width, height, r)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	svg, err := bc.GetSVG()
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, SVG: svg})
}

// handleDrawCanvas draws barcodes onto a gg canvas with scenic graphics
func handleDrawCanvas(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, apiResponse{Error: "POST only"})
		return
	}
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}
	width := getFormInt(r, "width", info.W)
	height := getFormInt(r, "height", info.H)
	x := getFormInt(r, "x", 0)
	y := getFormInt(r, "y", 0)
	foreColor := r.FormValue("fore_color")
	backColor := r.FormValue("back_color")
	showText := r.FormValue("show_text") != "0"
	fontSize := getFormInt(r, "font_size", 10)

	b64, err := drawCanvasScene(typeID, code, x, y, width, height, foreColor, backColor, showText, fontSize, info)
	if err != nil {
		writeJSON(w, apiResponse{Error: err.Error()})
		return
	}
	writeJSON(w, apiResponse{OK: true, Base64: b64})
}

func drawCanvasScene(typeID, code string, posX, posY, width, height int, foreColor, backColor string, showText bool, fontSize int, info barcodeTypeDef) (string, error) {
	W, H := 780, 680
	dc := gg.NewContext(W, H)

	// --- Sunset gradient sky ---
	for i := 0; i < H; i++ {
		t := float64(i) / float64(H)
		r := int(180 - 120*t)
		g := int(100 + 60*t)
		b := int(60 + 140*t)
		dc.SetRGBA255(r, g, b, 255)
		dc.DrawLine(0, float64(i), float64(W), float64(i))
		dc.Stroke()
	}

	// --- Sun ---
	dc.SetRGBA255(255, 200, 50, 180)
	dc.DrawCircle(float64(W)-120, 80, 60)
	dc.Fill()
	dc.SetRGBA255(255, 220, 100, 100)
	dc.DrawCircle(float64(W)-120, 80, 90)
	dc.Fill()

	// --- Clouds ---
	rng := rand.New(rand.NewSource(42))
	for i := 0; i < 5; i++ {
		cx := float64(rng.Intn(W - 100) + 50)
		cy := float64(rng.Intn(150) + 30)
		dc.SetRGBA255(255, 255, 255, 120+rng.Intn(60))
		dc.DrawEllipse(cx, cy, 50+float64(rng.Intn(30)), 15+float64(rng.Intn(10)))
		dc.Fill()
		dc.DrawEllipse(cx+30, cy-5, 40, 12)
		dc.Fill()
	}

	// --- Green grass ---
	grassY := float64(H) * 0.72
	dc.SetRGBA255(80, 160, 60, 255)
	dc.DrawRectangle(0, grassY, float64(W), float64(H)-grassY)
	dc.Fill()

	// --- Flowers ---
	for i := 0; i < 20; i++ {
		fx := float64(rng.Intn(W))
		fy := grassY + float64(rng.Intn(int(float64(H)-grassY)))
		sz := 3 + float64(rng.Intn(4))
		colors := []color.RGBA{
			{255, 100, 100, 200}, {255, 200, 50, 200}, {200, 100, 255, 200}, {255, 150, 200, 200},
		}
		dc.SetColor(colors[rng.Intn(len(colors))])
		dc.DrawCircle(fx, fy, sz)
		dc.Fill()
	}

	// --- Walking elephant ---
	drawElephant(dc, 100, grassY-40, 1.0)

	// --- Flying elephant (small) ---
	drawFlyingElephant(dc, float64(W)-200, 160, 0.6)

	// --- Barcode ---
	renderer := barcodeGg.New()
	defer renderer.Close()

	bc := createBarcodeForCanvas(typeID, barcode.FormatPNG, showText)
	bx := float64(posX)
	by := float64(posY)
	if bx == 0 && by == 0 {
		bx = float64(W)/2 - float64(width)/2
		by = grassY - float64(height) - 30
	}

	// White background card for barcode
	dc.SetRGBA255(255, 255, 255, 220)
	dc.DrawRoundedRectangle(bx-10, by-10, float64(width)+20, float64(height)+20, 8)
	dc.Fill()

	if err := renderer.DrawBarcode(dc, bc, code, bx, by, float64(width), float64(height)); err != nil {
		return "", err
	}

	// --- Coordinate label ---
	dc.SetRGBA255(0, 0, 0, 200)
	dc.DrawStringAnchored(fmt.Sprintf("(%d, %d) %dx%d", posX, posY, width, height), 10, float64(H)-16, 0, 0.5)

	// --- Title ---
	dc.SetRGBA255(255, 255, 255, 230)
	dc.DrawRoundedRectangle(8, 8, 260, 30, 4)
	dc.Fill()
	dc.SetRGBA255(30, 64, 175, 255)
	dc.DrawStringAnchored("Barcode.Go Canvas Demo", 16, 23, 0, 0.5)

	// Encode
	img := dc.Image()
	var buf bytes.Buffer
	if err := png.Encode(&buf, img); err != nil {
		return "", err
	}
	b64 := base64.StdEncoding.EncodeToString(buf.Bytes())
	return "data:image/png;base64," + b64, nil
}

func createBarcodeForCanvas(typeID, format string, showText bool) drawableBarcode {
	switch typeID {
	case "QR":
		return barcode.NewQRCode(format)
	case "DataMatrix":
		return barcode.NewDataMatrix(format)
	case "PDF417":
		return barcode.NewPDF417(format)
	case "Code128":
		bc := barcode.NewCode128(format)
		bc.ShowText = showText
		return bc
	case "Code39":
		bc := barcode.NewCode39(format)
		bc.ShowText = showText
		return bc
	case "Code93":
		bc := barcode.NewCode93(format)
		bc.ShowText = showText
		return bc
	case "NW7":
		bc := barcode.NewNW7(format)
		bc.ShowText = showText
		return bc
	case "ITF":
		bc := barcode.NewITF(format)
		bc.ShowText = showText
		return bc
	case "Matrix2of5":
		bc := barcode.NewMatrix2of5(format)
		bc.ShowText = showText
		return bc
	case "NEC2of5":
		bc := barcode.NewNEC2of5(format)
		bc.ShowText = showText
		return bc
	case "JAN13":
		bc := barcode.NewJAN13(format)
		bc.ShowText = showText
		return bc
	case "JAN8":
		bc := barcode.NewJAN8(format)
		bc.ShowText = showText
		return bc
	case "UPCA":
		bc := barcode.NewUPCA(format)
		bc.ShowText = showText
		return bc
	case "UPCE":
		bc := barcode.NewUPCE(format)
		bc.ShowText = showText
		return bc
	case "GS1_128":
		bc := barcode.NewGS1128(format)
		bc.ShowText = showText
		return bc
	case "GS1DataBar14":
		return barcode.NewGS1DataBar14(format, barcode.Omnidirectional)
	case "GS1DataBarLimited":
		return barcode.NewGS1DataBarLimited(format)
	case "GS1DataBarExpanded":
		return barcode.NewGS1DataBarExpanded(format, barcode.Unstacked, 2)
	case "YubinCustomer":
		return barcode.NewYubinCustomer(format)
	default:
		return barcode.NewQRCode(format)
	}
}

func drawElephant(dc *gg.Context, ex, ey, s float64) {
	// Body
	dc.SetRGBA255(140, 155, 175, 230)
	dc.DrawEllipse(ex, ey, 50*s, 30*s)
	dc.Fill()
	// Head
	dc.DrawEllipse(ex+45*s, ey-20*s, 25*s, 22*s)
	dc.Fill()
	// Ear
	dc.SetRGBA255(120, 135, 160, 200)
	dc.DrawEllipse(ex+60*s, ey-30*s, 18*s, 20*s)
	dc.Fill()
	// Eye
	dc.SetRGBA255(255, 255, 255, 255)
	dc.DrawCircle(ex+45*s, ey-28*s, 4*s)
	dc.Fill()
	dc.SetRGBA255(20, 20, 40, 255)
	dc.DrawCircle(ex+46*s, ey-27*s, 2*s)
	dc.Fill()
	// Trunk
	dc.SetRGBA255(130, 145, 170, 255)
	dc.SetLineWidth(6 * s)
	dc.MoveTo(ex+65*s, ey-10*s)
	dc.LineTo(ex+78*s, ey+5*s)
	dc.LineTo(ex+72*s, ey+25*s)
	dc.LineTo(ex+60*s, ey+30*s)
	dc.Stroke()
	dc.SetLineWidth(1)
	// Legs
	dc.SetRGBA255(130, 145, 170, 220)
	for _, lx := range []float64{-25, -5, 20, 40} {
		dc.DrawRectangle(ex+lx*s, ey+25*s, 14*s, 28*s)
		dc.Fill()
	}
	// Tail
	dc.SetRGBA255(130, 145, 170, 255)
	dc.SetLineWidth(3 * s)
	dc.DrawArc(ex-55*s, ey, 12*s, math.Pi*0.6, math.Pi*1.4)
	dc.Stroke()
	dc.SetLineWidth(1)
}

func drawFlyingElephant(dc *gg.Context, ex, ey, s float64) {
	// Body
	dc.SetRGBA255(160, 175, 200, 200)
	dc.DrawEllipse(ex, ey, 35*s, 20*s)
	dc.Fill()
	// Head
	dc.DrawEllipse(ex+30*s, ey-10*s, 18*s, 16*s)
	dc.Fill()
	// Wings
	dc.SetRGBA255(200, 210, 240, 160)
	dc.DrawEllipse(ex-10*s, ey-25*s, 25*s, 12*s)
	dc.Fill()
	dc.DrawEllipse(ex+10*s, ey-28*s, 25*s, 12*s)
	dc.Fill()
	// Eye
	dc.SetRGBA255(255, 255, 255, 255)
	dc.DrawCircle(ex+32*s, ey-15*s, 3*s)
	dc.Fill()
	dc.SetRGBA255(20, 20, 40, 255)
	dc.DrawCircle(ex+33*s, ey-14*s, 1.5*s)
	dc.Fill()
	// Trunk
	dc.SetRGBA255(150, 165, 190, 255)
	dc.SetLineWidth(4 * s)
	dc.MoveTo(ex+45*s, ey-5*s)
	dc.LineTo(ex+55*s, ey+5*s)
	dc.LineTo(ex+50*s, ey+15*s)
	dc.Stroke()
	dc.SetLineWidth(1)
}

// handlePDF generates a report PDF with barcode samples
func handlePDF(w http.ResponseWriter, r *http.Request) {
	r.ParseForm()
	typeID := r.FormValue("type")
	if typeID == "" {
		typeID = "QR"
	}
	code := r.FormValue("code")
	info, ok := barcodeMap[typeID]
	if !ok {
		info = barcodeMap["QR"]
		typeID = "QR"
	}
	if code == "" {
		code = info.Default
	}

	pdfBytes, err := generateReportPDF(typeID, code, info)
	if err != nil {
		http.Error(w, err.Error(), 500)
		return
	}
	w.Header().Set("Content-Type", "application/pdf")
	w.Header().Set("Content-Disposition", "inline; filename=barcode_report.pdf")
	w.Write(pdfBytes)
}

func generateReportPDF(typeID, code string, info barcodeTypeDef) ([]byte, error) {
	pdf := &gopdf.GoPdf{}
	pdf.Start(gopdf.Config{PageSize: *gopdf.PageSizeA4})
	pdf.AddPage()
	loadFont(pdf)

	pw, ph := gopdf.PageSizeA4.W, gopdf.PageSizeA4.H

	renderer := barcodeGopdf.New(pdf)
	defer renderer.Close()

	today := time.Now().Format("2006-01-02")

	// ========== Header (blue gradient) ==========
	pdf.SetFillColor(30, 64, 175)
	pdf.RectFromUpperLeftWithStyle(0, 0, pw, 70, "F")
	pdf.SetFillColor(124, 58, 237)
	pdf.RectFromUpperLeftWithStyle(0, 70, pw, 4, "F")

	pdf.SetTextColor(255, 255, 255)
	pdf.SetFontSize(20)
	pdf.SetX(40)
	pdf.SetY(18)
	pdf.Cell(nil, "BARCODE REPORT")

	pdf.SetFontSize(9)
	pdf.SetX(40)
	pdf.SetY(42)
	pdf.Cell(nil, fmt.Sprintf("%s | %s", info.Label, typeID))
	pdf.SetX(pw - 120)
	pdf.SetY(18)
	pdf.Cell(nil, today)
	pdf.SetX(pw - 120)
	pdf.SetY(32)
	pdf.Cell(nil, "Pao@Office")
	pdf.SetX(pw - 120)
	pdf.SetY(46)
	pdf.Cell(nil, "pao.ac")

	// ========== Data section ==========
	pdf.SetTextColor(71, 85, 105)
	pdf.SetFontSize(9)
	pdf.SetX(40)
	pdf.SetY(90)
	pdf.Cell(nil, fmt.Sprintf("Data: %s", code))

	// ========== Main barcode ==========
	y := 110.0
	is2d := info.Dim == "2d"
	var bw, bh float64
	if is2d {
		bw, bh = 200, 200
		if typeID == "PDF417" {
			bw, bh = 300, 120
		}
	} else if info.Dim == "postal" {
		bw, bh = 300, 40
	} else {
		bw, bh = 300, 120
	}

	bc := createBarcodeForCanvas(typeID, barcode.FormatPNG, true)
	if err := renderer.DrawBarcode(bc, code, 40, y, bw, bh); err != nil {
		return nil, fmt.Errorf("main barcode: %w", err)
	}

	// ========== Sample table ==========
	y += bh + 30
	pdf.SetFillColor(30, 64, 175)
	pdf.RectFromUpperLeftWithStyle(40, y, pw-80, 20, "F")
	pdf.SetTextColor(255, 255, 255)
	pdf.SetFontSize(8)
	pdf.SetX(50)
	pdf.SetY(y + 5)
	pdf.Cell(nil, "Barcode")
	pdf.SetX(300)
	pdf.SetY(y + 5)
	pdf.Cell(nil, "Data")
	y += 20

	samples := getSampleCodes(typeID)
	for i, samp := range samples {
		ry := y + float64(i)*55
		if i%2 == 0 {
			pdf.SetFillColor(248, 250, 252)
			pdf.RectFromUpperLeftWithStyle(40, ry, pw-80, 55, "F")
		}

		sbc := createBarcodeForCanvas(typeID, barcode.FormatPNG, false)
		var sw, sh float64
		if is2d {
			sw, sh = 42, 42
		} else if info.Dim == "postal" {
			sw, sh = 180, 25
		} else {
			sw, sh = 190, 32
		}
		renderer.DrawBarcode(sbc, samp, 50, ry+5, sw, sh)

		pdf.SetTextColor(0, 0, 0)
		pdf.SetFontSize(8)
		pdf.SetX(300)
		pdf.SetY(ry + 20)
		pdf.Cell(nil, samp)
	}

	// ========== Footer ==========
	pdf.SetStrokeColor(226, 232, 240)
	pdf.SetLineWidth(0.5)
	pdf.Line(40, ph-40, pw-40, ph-40)
	pdf.SetTextColor(148, 163, 184)
	pdf.SetFontSize(7)
	pdf.SetX(40)
	pdf.SetY(ph - 30)
	pdf.Cell(nil, "Generated by Barcode.Go — barcode-go + GoPdfRenderer")
	pdf.SetX(pw - 60)
	pdf.SetY(ph - 30)
	pdf.Cell(nil, "pao.ac")

	var buf bytes.Buffer
	_, err := pdf.WriteTo(&buf)
	if err != nil {
		return nil, err
	}
	return buf.Bytes(), nil
}

func getSampleCodes(typeID string) []string {
	switch typeID {
	case "QR":
		return []string{"Hello World", "https://www.pao.ac", "SAMPLE-001", "TEST DATA"}
	case "DataMatrix":
		return []string{"DataMatrix-1", "ABCDE12345", "Test-DM", "PAO"}
	case "PDF417":
		return []string{"PDF417-Test", "ABCDEFGHIJ", "12345", "PAO"}
	case "Code128":
		return []string{"Hello-123", "ABC-2026", "Test-Code128", "BARCODE"}
	case "Code39":
		return []string{"HELLO-123", "ABC-2026", "TEST", "CODE39"}
	case "Code93":
		return []string{"CODE93", "TEST-93", "ABC123", "SAMPLE"}
	case "NW7":
		return []string{"A123456B", "A999999A", "B12345B", "C00001D"}
	case "ITF":
		return []string{"123456", "000000", "111111", "999999"}
	case "Matrix2of5":
		return []string{"1234", "5678", "0000", "9999"}
	case "NEC2of5":
		return []string{"1234", "5678", "0000", "9999"}
	case "JAN13":
		return []string{"4912345123459", "4901234567890", "4567890123456", "1234567890128"}
	case "JAN8":
		return []string{"1234567", "0000000", "9999999", "4567890"}
	case "UPCA":
		return []string{"01234567890", "12345678901", "99999999999", "00000000000"}
	case "UPCE":
		return []string{"0123456", "0000000", "0999999", "0123450"}
	case "GS1_128":
		return []string{"[01]04912345123459", "[01]09501234567891", "[01]04567890123450", "[01]01234567890128"}
	case "GS1DataBar14":
		return []string{"0123456789012", "9999999999999", "0000000000000", "1234567890123"}
	case "GS1DataBarLimited":
		return []string{"0123456789012", "0100000000000", "0999999999999", "0123456789012"}
	case "GS1DataBarExpanded":
		return []string{"[01]90012345678908", "[01]95012345678903", "[01]98012345678902", "[01]90000000000003"}
	case "YubinCustomer":
		return []string{"1060032", "1000001", "5300001", "6008799"}
	default:
		return []string{"TEST1", "TEST2", "TEST3", "TEST4"}
	}
}
